<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\Artisan;
use Inside\Content\Models\Contents\Sites;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;

final class UpdateContentTypes extends Migration
{
    use CreateContentTypes;

    public function up(): void
    {
        $this->bootUp(__FILE__);

        $this->migrateContent('forms');

        $this->getBridge()?->contentTypeDelete('movements_categories');
        $this->getBridge()?->contentTypeDelete('movements');

        $this->getBridge()?->contentTypeUpdate('dms_link', [
            'title' => [
                'fr' => 'Lien',
                'de' => 'Link',
            ],
        ]);

        $this->getBridge()?->updateContentTypeOptions('sites', [
            'searchable' => true,
            'listing_type' => 'content',
        ]);

        $this->getBridge()?->contentTypeCreateOrUpdateField(
            'users',
            'address',
            [
                'name' => 'address',
                'searchable' => true,
                'translatable' => false,
                'search_result_field' => true,
                'searchable_filter' => false,
                'required' => false,
                'title' => [
                    'fr' => 'Adresse',
                    'de' => 'Adresse',
                ],
                'widget' => [
                    'type' => BridgeContentType::STRING_WIDGET,
                    'settings' => [
                        'size' => 60,
                        'placeholder' => '',
                    ],
                ],
                'type' => BridgeContentType::TEXT_FIELD,
                'settings' => [
                    'max_length' => 255,
                    'is_ascii' => false,
                    'case_sensitive' => false,
                    'cardinality' => 1,
                ],
            ],
        );

        collect([
            'firstname',
            'lastname',
            'phone',
            'mobile_phone',
            'function',
        ])->each(fn (string $searchableField) => $this->getBridge()?->contentTypeUpdateField('users', $searchableField, ['searchable' => true]));

        $this->translateSites();

        Artisan::call('inside:paragraph:fix:labels');

        $this->finishUp();
    }

    private function translateSites(): void
    {
        $locale = config('app.locale');
        $languages = collect(list_languages())->diff([$locale]);

        $bridgeContent = new BridgeContent();

        $originals = Sites::where('langcode', $locale)->get();

        $languages->each(function (string $language) use ($originals, $bridgeContent) {
            $existing = Sites::query()
                ->whereIn('uuid_host', $originals->pluck('uuid_host'))
                ->where('langcode', $language)
                ->pluck('uuid_host');

            $originals
                ->reject(fn (Sites $site) => $existing->contains($site->uuid_host))
                ->each(fn (Sites $original) => $bridgeContent->contentInsert($original->content_type, [
                    'title' => $original->title,
                    'uuid_host' => $original->uuid_host,
                    'status' => $original->status,
                    'langcode' => $language,
                ]));
        });
    }
}
