<?php

namespace Inside\WEB2\Console;

use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Console\Command;
use Inside\Content\Models\Contents\News;
use Inside\Content\Models\Contents\NewsCategories;
use Inside\Host\Bridge\BridgeContent;
use Inside\RSS\Helpers\RssFeeder;

class ImportRSSNews extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'web2:import-rss';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Importer les actualités du flux RSS';

    protected BridgeContent $bridgeContent;

    /**
     * InsideImport constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->bridgeContent = new BridgeContent();
    }

    public function handle(): void
    {
        // lire le flux RSS
        $feeder = new RssFeeder(config('importRSS.webedia_rss'), config('importRSS.fields_to_match'));
        $rssNews = collect($feeder->feed());

        // recupéréer les actualités en base qui sont de nature importée par flux rss
        $bddNews = News::whereNotNull('webedia_guid')->orderBy('published_at', 'desc')->get();

        // sur la première entrée des news bdd, récupèrer la date de publication comme critère de comparaison
        $latestInsertedDate = $bddNews->first()->published_at;

        $newsToInsert = $rssNews->filter(fn ($item) => $item['pubDate'] > $latestInsertedDate);

        // pour l'insertion on inverse en premier lieu, car en cas de plantage on ne pourra plus récupérer celles qui devaient l'être (vu qu'elles seront toujours inférieur à la date de la dernière insertion)
        $newsToInsert = $newsToInsert->reverse();
        $newsToInsert->each(function ($item) {
            $prepared = $this->prepareNews($item);
            $this->bridgeContent->contentInsert('news', $prepared);
        });
    }

    public function prepareNews(array $item): array
    {
        if (empty($item['enclosure'])) {
            $item['enclosure'] = back_path('resources/assets/img/logo.png');
        }
        $imagePath = $this->saveImage($item['enclosure']);
        $data = [
            'bundle' => 'news',
            'type' => 'node',
            'title' => $item['title'],
            'image' => $imagePath,
            'chapo' => $item['description'],
            'content' =>
                [
                    [
                        'bundle' => 'text',
                        'body' => '<p class="ds-base-text-container">' . trim($item['content']) . '</p>',
                    ],
                ],
            'news_categories' => $this->getNewsCategory($item['category']),
            'webedia_guid' => $this->getGUID($item['guid']),
            'published_at' => Carbon::parse($item['pubDate'])->format('Y-m-d H:i:s'),
            'status' => 1,
            'is_featured' => false,
        ];
        return $data;
    }

    public function getNewsCategory(string $categoryTofind): ?string
    {
        $category = NewsCategories::where('title', $categoryTofind)->first();
        if ($category) {
            return $category->uuid;
        }
        return $this->bridgeContent->contentInsert('news_categories', [
            'title' => $categoryTofind,
            'status' => true,
        ]);
    }

    public function saveImage(string $imageUrl): string|null
    {
        $image = file_get_contents($imageUrl);
        if (!$image) {
            Log::error(' error occurred while trying to get rss image. Url: ' . $imageUrl);
            return null;
        }
        $size = getimagesize($imageUrl);
        if (!$size) {
            Log::error(' error occurred while trying to get rss image size. Url: ' . $imageUrl);
            return null;
        }
        $extension = image_type_to_extension($size[2]);
        $chunkId = Str::random(32);
        $time = time();
        $chunkPath = "chunks/$time/$chunkId$extension";
        Storage::makeDirectory("chunks/$time");
        Storage::disk('local')->put($chunkPath, $image);
        return $chunkPath;
    }

    public function getGUID(string|int $newsLink): int
    {
        if (is_int($newsLink)) {
            return $newsLink;
        }
        preg_match('#/\?p\=([0-9]+)$#', $newsLink, $guid);
        return (int)$guid[1];
    }
}
