import { http } from '@inside-library/helpers/Http'
import { insertIf } from '@inside-library/helpers/Arrays'

import { normalizeColor } from '../../helpers'

import {
  capsuleNewsFields,
  categoriesFieldsWithSubCategories
} from '../../constants/news-fields'

const fetchBuSpace = async ({ $route }) => {
  /**
   * Fetch current Business Unit
   */

  try {
    const { data: bu } = await http.get('content/business_units', {
      fields: [
        'uuid',
        'title',
        'banner',
        'introduction',
        'admin',
        'content_type',
        {
          favorites_categories: [
            'uuid',
            'title',
            'content_type',
            { colors: ['color'] }
          ]
        },
        {
          trends_categories: [
            'uuid',
            'title',
            'content_type',
            { colors: ['color'] }
          ]
        },
        {
          colors: ['uuid', 'color']
        },
        {
          sales_managers: [
            'uuid',
            'lastname',
            'firstname',
            'email',
            'function',
            'phone',
            'image',
            'name'
          ]
        },
        {
          bu_highlights: {
            fields: [
              'uuid',
              'title',
              'content_type',
              'admin',
              'image',
              'cta_title',
              'cta_file',
              'cta_link',
              { iframe_pages: ['uuid', 'title', 'slug'] },
              { page_catalogs: ['uuid', 'title', 'slug'] },
              { custom: ['uuid', 'title', 'slug'] }
            ],
            filters: {
              reverse: true,
              'status:eq': 1,
              sort: 'weight:asc'
            }
          }
        }
      ],
      filters: {
        'slug:eq': $route.params.slug
      }
    })

    /**
     * Fetch Favorites and Trends from the current Business Unit
     */

    const favoritesBuCategoryUuid = bu[0].favorites_categories.uuid
    const trendsBuCategoryUuid = bu[0].trends_categories.uuid

    /** Favorites */
    const favoritesPromise = http.get('content/favorites', {
      filters: {
        categories: { 'uuid:eq': favoritesBuCategoryUuid },
        'status:eq': 1,
        sort: 'published_at:desc',
        limit: 4,
        'published_at:lte': 'now()'
      },
      fields: [
        ...capsuleNewsFields,
        { categories: categoriesFieldsWithSubCategories }
      ]
    })

    /** Trends */
    const { data: trendsCategories } = await http.get(
      'content/trends_categories',
      {
        fields: [
          {
            trends_categories: {
              fields: ['uuid', 'title'],
              filters: { reverse: true }
            }
          }
        ],
        filters: { 'uuid:eq': trendsBuCategoryUuid }
      }
    )

    const trendsSubCategoriesUuids =
      trendsCategories[0] && trendsCategories[0].trendsCategories
        ? trendsCategories[0].trends_categories.data.map(({ uuid }) => uuid)
        : []

    const trendsPromise = http.get('content/trends', {
      filters: {
        $or: [
          { categories: { 'uuid:eq': trendsBuCategoryUuid } },
          ...insertIf(!!trendsSubCategoriesUuids.length, {
            categories: { 'uuid:in': trendsSubCategoriesUuids }
          })
        ],
        'status:eq': 1,
        sort: 'published_at:desc',
        limit: 4,
        'published_at:lte': 'now()'
      },
      fields: [
        ...capsuleNewsFields,
        { categories: categoriesFieldsWithSubCategories }
      ]
    })

    const [{ data: favorites }, { data: trends }] = await Promise.all([
      favoritesPromise,
      trendsPromise
    ])

    const news = normalizeColor([...favorites, ...trends])

    return {
      bu: bu[0],
      news: news.sort((a, b) => b.published_at - a.published_at).slice(0, 4)
    }
  } catch (err) {
    throw new Error(err)
  }
}

export { fetchBuSpace }
