<?php

namespace Inside\UNIP\Console;

use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Host\Bridge\BridgeContent;
use Inside\UNIP\Facades\ImportCsv;
use League\Csv\Reader;
use Symfony\Component\Console\Helper\ProgressBar;

class ImportCsvCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'unip:catalogs:import';

    /**
     * @var string
     */
    protected $signature = 'unip:catalogs:import {type} {--d|delete-before} {--s|single} {--l|language=}';

    /**
     * @var string
     */
    protected $description = 'Import catalogs';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * @throws \League\Csv\Exception
     * @throws Exception
     */
    public function handle(): void
    {
        /** @var string $type */
        $type = $this->argument('type');
        /** @var string $langcode */
        $langcode = !empty($this->option('language')) ? $this->option('language') : 'fr';
        $authorizeTypes = config('catalogs.authorize_type');

        if (!in_array($type, $authorizeTypes)) {
            throw new Exception('The type of the '.$type.' catalog is not good it must be either nutra, cos or pharma');
        }

        if ($this->option('delete-before')) {
            $this->writeln('<comment>Deleting current catalog contents before import</comment>');

            $this->call('inside:content:delete', ['type' => 'catalogs_' . $type, '--langcode' => $langcode]);
        }

        $this->writeln("<comment>Début de l'import du catalogue $type !</comment>");

        $filePaths = [];

        if ($type === 'cos' && !$this->option('single')) {
            $files = Storage::disk('local')->files('csv/cos');

            foreach ($files as $file) {
                /** @var array $info */
                $info = pathinfo($file);
                if ($info['extension'] !== 'csv') {
                    continue;
                }

                $filePaths[] = __DIR__ .'/../../../../../storage/app/' . $file;
            }
        } else {
            $filePath = __DIR__ .'/../../../../../storage/app/csv/'.$type.'.csv';
            $filePaths = [$filePath];
        }

        foreach ($filePaths as $filePath) {
            $records = $this->getCsvRecords($filePath);
            $this->writeln(sprintf('<info>Il y a [%d] catalogues %s à importer dans le csv %s</info>', count($records), $type, pathinfo($filePath)['basename']));

            $this->buildCatalogs($records, $type, $langcode);
        }
    }

    /**
     * We take the csv file and put it in an array with the right keys and values
     * @throws \League\Csv\Exception
     */
    protected function getCsvRecords(string $filePath): array
    {
        $csv = Reader::createFromPath($filePath, 'r');
        $encoding = mb_detect_encoding($csv->getContent(), mb_list_encodings(), true);

        if ($encoding !== 'UTF8') {
            $csv->setOutputBOM(Reader::BOM_UTF8);
            $csv->addStreamFilter('convert.iconv.' . $encoding . '/UTF-8');
        }

        $csv->setDelimiter(';');
        $csv->setHeaderOffset(0);

        $headers = array_map(
            function ($item) {
                return Str::lower(trim($item));
            },
            $csv->getHeader()
        );

        return iterator_to_array($csv->getRecords($headers));
    }

    /**
     * Creation of the right table for the catalog
     *
     * @param array $records
     * @param string $type
     * @param string $langcode
     * @throws Exception
     */
    protected function buildCatalogs(array $records, string $type, string $langcode): void
    {
        $value = config('catalogs.' . $type);
        $extras = config('catalogs.extra.' . $langcode . '.' . $type);
        if ($extras) {
            $value = array_merge($value, $extras);
        }

        $bar = $this->getOutput()->createProgressBar(count($records));
        $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");
        foreach ($records as $record) {
            $data = [];
            $valueReference = ImportCsv::getReferenceValue($record);

            foreach ($value as $modelField => $fileField) {
                if (($fileField instanceof \Closure)) {
                    $data[$modelField] = $fileField($record, $langcode, $valueReference);
                } else {
                    $data[$modelField] = $record[$fileField];
                }
            }
            if (empty($data['title'])) {
                continue;
            }
            $data['langcode'] = $langcode;

            $this->insertCatalogs($data, $bar);
        }
    }

    /**
     * Insert catalogs in the database
     *
     * @param array $data
     * @param ProgressBar $bar
     * @throws Exception
     */
    protected function insertCatalogs(array $data, ProgressBar $bar): void
    {
        $bridge = new BridgeContent();

        $query = call_user_func(type_to_class($data['bundle']) . '::query');
        if ($data['bundle'] === 'catalogs_cos') {
            if (!empty($data['functions'])) {
                $query->whereHas('functions', function ($query) use ($data) {
                    $query->whereIn('inside_content_functions.uuid', $data['functions']);
                });
            }
        } elseif ($data['bundle'] === 'catalogs_pharma') {
            if (!empty($data['trade_name'])) {
                $query->where('trade_name', $data['trade_name']);
            }
        }

        $langcode = !empty($this->option('language')) ? $this->option('language') : 'fr';
        if (!$query->where('title', $data['title'])->where('langcode', $langcode)->exists()) {
            $bar->setMessage('Importation du catalogue => ' . $data['title'] . '');
            $bridge->contentInsert($data['bundle'], $data);
            $bar->advance();
        }
    }

    /**
     * Write
     *
     * @param  string  $message
     */
    protected function write(string $message): void
    {
        $this->getOutput()->write($message);
    }

    /**
     * Write and return to new line
     *
     * @param  string  $message
     */
    protected function writeln(string $message): void
    {
        $this->getOutput()->writeln($message);
    }

    /**
     * Write a result
     *
     * @param  bool  $success
     */
    protected function writeResult(bool $success): void
    {
        if ($success) {
            $this->writeln(' <fg=green>✔</fg=green>');

            return;
        }
        $this->writeln(' <fg=red>✘</fg=red>');
    }
}
