<?php

namespace Inside\UNIP\Services;

use Exception;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\BusinessUnits;
use Inside\Content\Models\Model;
use Inside\Host\Bridge\BridgeContent;

/**
 * Class ImportCsvService
 * @package Inside\UNIP\Services
 */
class ImportCsvService
{
    public function __construct(
        protected BridgeContent $bridge
    ) {
    }

    /**
     * Takes the name of the fields that are filled with a 'x' for the relations
     */
    public function getReferenceValue(array $record): array
    {
        $valueReference = [];
        foreach ($record as $key => $value) {
            //todo handle english version here
            if (Str::lower($value) === 'x' || is_numeric($value)) {
                if (is_numeric($value) && $key !== 'naturality') {
                    continue;
                }
                switch ($key) {
                    case 'rspo mb':
                        $key = 'RSPO MB';
                        break;
                    case 'approuvé cosmos':
                        $key = 'approuvé COSMOS';
                        break;
                    case 'mono inci':
                        $key = 'mono INCI';
                        break;
                    case 'non-ogm':
                        $key = 'non-OGM';
                        break;
                    case 'certifié cosmos (bio)':
                        $key = 'certifié COSMOS (bio)';
                        break;
                }
                array_push($valueReference, $key);
            }
        }

        return $valueReference;
    }

    /**
     * Look if the references already exist and return the uuids otherwise it creates the contents and returns the uuids too
     * @throws Exception
     */
    public function getEntityReferences(
        array $record,
        string $bundle,
        string $column,
        string $langcode,
        bool $referenceImage,
        string $separator = "\n"
    ): array {
        $query = call_user_func(type_to_class($bundle).'::get');
        if (Str::contains($record[$column], $separator)) {
            /** @phpstan-ignore-next-line */
            $column = explode($separator, $record[$column]);
        }
        $dataUuids = [];
        if (is_array($column)) {
            foreach ($column as $name) {
                $name = Str::ucfirst(trim($name));
                if (empty($name)) {
                    continue;
                }
                if ($bundle === "properties") {
                    $valueProperties = explode("\n", $name);
                    foreach ($valueProperties as $property) {
                        $dataUuids = $this->getUuidReference(
                            $dataUuids,
                            $query,
                            $property,
                            $bundle,
                            $langcode,
                            $referenceImage
                        );
                    }
                } else {
                    $dataUuids = $this->getUuidReference($dataUuids, $query, $name, $bundle, $langcode, $referenceImage);
                }
            }

            return $dataUuids;
        } else {
            $name = Str::ucfirst(trim($record[$column]));
            if (empty($name)) {
                return [];
            }

            return $this->getUuidReference($dataUuids, $query, $name, $bundle, $langcode, $referenceImage);
        }
    }


    /**
     * @throws Exception
     */
    protected function getUuidReference(
        array $dataUuid,
        Builder $query,
        string $name,
        string $bundle,
        string $langcode,
        bool $referenceImage
    ): array {
        /** @var Content|null $entity */
        $entity = $query->where('title', $name)->where('langcode', $langcode)->first();
        if ($entity) {
            $dataUuid[] = $entity->uuid;
        } else {
            $image = $referenceImage ? $this->getAndUploadImageAndFile($name, $referenceImage) : null;
            $dataUuid[] = $this->contentInsertReference($name, $bundle, $langcode, $image);
        }

        return $dataUuid;
    }

    /**
     * Upload image and file for the catalogs
     */
    public function getAndUploadImageAndFile(string $fileName, bool $referenceImage): ?string
    {
        $imgPath = "csv/files/$fileName";
        if ($referenceImage) {
            $fileName = $fileName.'.png';
            $imgPath = "csv/files/$fileName";
        }
        if (!Storage::exists($imgPath)) {
            return null;
        }
        $chunkId = Str::random(32);
        $temporaryFilePath = "chunks/$chunkId";
        $finalPath = "$temporaryFilePath/$fileName";
        Storage::copy($imgPath, $finalPath);

        return $finalPath;
    }

    /**
     * Upload files in a paragraph
     */
    public function manageParagraphForData(string $record, array $paragraphMultiField = []): array
    {
        $contents = [];
        if (!empty($paragraphMultiField)) {
            foreach ($paragraphMultiField as $key => $recordMulti) {
                if (empty($recordMulti)) {
                    continue;
                }
                $contents[] = ['bundle' => 'text', 'body' => '<h4>'.Str::ucfirst($key).'</h4>'];
                $contents = $this->manageUploadFileParagraph($recordMulti, $contents);
            }
        } else {
            $contents = $this->manageUploadFileParagraph($record, $contents);
        }

        return $contents;
    }

    /**
     */
    protected function manageUploadFileParagraph(string $records, array $contents): array
    {
        $paragraphFileNames = explode("\n", $records);
        if (is_array($paragraphFileNames)) {
            foreach ($paragraphFileNames as $paragraphFileName) {
                $fileTitle = explode(".", trim($paragraphFileName));
                $file = $this->getAndUploadImageAndFile(trim($paragraphFileName), false);
                if ($file) {
                    $contents[] = [
                        'bundle' => 'file',
                        'file' => $file,
                        'title' => Str::ucfirst($fileTitle[0]),
                    ];
                }
            }
        }

        return $contents;
    }

    /**
     * Builds the array and inserts the content type in DB
     * @throws Exception
     */
    protected function contentInsertReference(
        string $name,
        string $bundle,
        string $langcode,
        string $image = null
    ): string {
        return (string) $this->bridge->contentInsert($bundle, [
            'langcode' => $langcode,
            'title' => $name,
            'type' => 'node',
            'bundle' => $bundle,
            'image' => $image,
        ]);
    }

    /**
     * @throws Exception
     */
    public function setBusinessUnit(string $title, string $langcode): string
    {
        $businessUnit = BusinessUnits::query()->where('title', $title)->where('langcode', $langcode)->first();
        if ($businessUnit) {
            return $businessUnit->uuid;
        }

        return (string) $this->bridge->contentInsert('business_units', [
            'langcode' => $langcode,
            'title' => $title,
            'type' => 'node',
            'bundle' => 'business_units',
        ]);
    }
}
