<?php

namespace Inside\TRYB\Services;

use Carbon\Carbon;
use Illuminate\Contracts\Routing\UrlRoutable;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\InteractsWithTime;
use Inside\TRYB\Mail\PrepareRegistration;

/**
 * Class Registration
 *
 * @package Inside\TRYB\Services
 */
class Registration
{
    use InteractsWithTime;

    /**
     * Sign an uri
     */
    public function signedUri(string $uri, array $parameters = [], Carbon $expiration = null): string
    {
        $url = (array) parse_url($uri);
        if (isset($url['query'])) {
            parse_str($url['query'], $params);
        } else {
            $params = [];
        }

        $parameters = $this->formatParameters($params + $parameters);

        if ($expiration) {
            $parameters = $parameters + ['expires' => $this->availableAt($expiration)];
        }
        ksort($parameters);

        $key = config('app.key');

        return $this->buildUrl(
            $url,
            $parameters + [
                'signature' => hash_hmac('sha256', $this->buildUrl($url, $parameters), $key),
            ]
        );
    }

    /**
     * Validate a signature
     */
    public function hasValidSignature(string $uri): bool
    {
        $url = (array) parse_url($uri);
        if (isset($url['query'])) {
            parse_str($url['query'], $params);
        } else {
            $params = [];
        }

        $original = rtrim(
            $this->buildUrl($url).'?'.http_build_query(
                Arr::except($params, 'signature'),
                '',
                '&',
                PHP_QUERY_RFC3986
            ),
            '?'
        );

        $expires = Arr::get($params, 'expires');
        $key = config('app.key');
        $signature = hash_hmac('sha256', $original, $key);

        return hash_equals($signature, (string) Arr::get($params, 'signature', ''))
            && !($expires && Carbon::now()->getTimestamp() > $expires);
    }

    /**
     * Send an email with a signed url to the start-registration form
     */
    public function sendRegistrationMail(string $langCode, string $email): bool
    {
        // Build signed url
        $url = $this->signedUri(
            config('app.url').'/start-registration',
            ['email' => $email],
            Carbon::now()->addMinutes(config('app.registration.lifetime', 30))
        );

        Mail::to($email)->send(new PrepareRegistration($langCode, $url));

        return true;
    }

    /**
     * Format the array of URL parameters.
     */
    protected function formatParameters(mixed $parameters): array
    {
        $parameters = Arr::wrap($parameters);
        foreach ($parameters as $key => $parameter) {
            if ($parameter instanceof UrlRoutable) {
                $parameters[$key] = $parameter->getRouteKey();
            }
        }

        return $parameters;
    }

    /**
     * Build url
     */
    public function buildUrl(array $url, array $params = []): string
    {
        ksort($params, SORT_STRING);
        $url['query'] = http_build_query($params, '', '&');
        $scheme = isset($url['scheme']) ? $url['scheme'].'://' : '';
        $host = isset($url['host']) ? $url['host'] : '';
        $port = isset($url['port']) ? ':'.$url['port'] : '';
        $user = isset($url['user']) ? $url['user'] : '';
        $pass = isset($url['pass']) ? ':'.$url['pass'] : '';
        $pass = ($user || $pass) ? "$pass@" : '';
        $path = isset($url['path']) ? $url['path'] : '';
        $query = $url['query'] ? '?'.$url['query'] : '';
        $fragment = isset($url['fragment']) ? '#'.$url['fragment'] : '';

        return $scheme.$user.$pass.$host.$port.$path.$query.$fragment;
    }
}
