<?php

namespace Inside\TRYB\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Mail;
use Inside\Authentication\Models\User;
use Inside\Authentication\Services\AuthenticationService;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\TRYB\Facades\Registration;
use Inside\TRYB\Mail\AccountStatusChanged;
use Inside\TRYB\Mail\PendingAccount;
use Laravel\Lumen\Routing\Controller;

/**
 * Class GameController
 *
 * @OA\Tag(name="Enregistrement TRYB",description="
# Introduction

Il s'agit du service d'enregistrement d'utilisateur spécifique à l'instance de TRYB
")
 *
 * @package Inside\TRYB\Http\Controllers
 */
class RegistrationController extends Controller
{
    /**
     * Pre enregistrement
     *
     * @param Request $request
     * @return array
     */
    public function checkinRegistration(Request $request)
    {
        if (Auth::check()) {
            /** @var User $user */
            $user = Auth::user();

            // Check that user is connected in front has well
            if ($request->header('api-token') !== null && $request->header('api-token') === $user->currentAccessToken()->token) {
                return [
                    'redirect' => '/',
                ];
            } else {
                $service = new AuthenticationService();
                $service->logout($user);
            }
        }

        $this->validate(
            $request,
            [
                'email' => 'required|email',
            ]
        );
        $email = $request->get('email');
        $langCode = $request->get('langcode', list_languages()[0]);
        Lang::setLocale($langCode);

        // 3 Choices !

        // Default is unauthorized account
        $redirect = 'unauthorized-account';

        // First let's see if we have this email in database
        $user = Users::where('email', $email)->first();

        // Quid status 0 ?
        if ($user && ($user->is_from_ldap || $user->registration_status === 'accepted')) {
            // Redirect to page explaining how to reset password
            $redirect = 'account-already-exist';
        } else {
            $redirect = 'explain-registration';

            // Send email
            Registration::sendRegistrationMail($langCode, $email);
        }

        return [
            'redirect' => $redirect,
        ];
    }

    /**
     * Création du compte avec uniquement l'email et le mot de passe
     *
     * @param Request $request
     * @return JsonResponse
     * @throws Exception
     */
    public function createRegistration(Request $request): JsonResponse
    {
        $this->validate(
            $request,
            [
                'email' => 'required|email',
                'password' => config('app.password_rules', 'required|string|confirmed|min:6'),
                'signature' => [
                    'required',
                    'string',
                    function ($attribute, $value, $fail) use ($request) {
                        $uri = Registration::buildUrl(parse_url(config('app.url') . '/start-registration'), [
                            'email' => $request->get('email'),
                            'expires' => $request->get('expires'),
                            'signature' => $value,
                        ]);
                        if (!Registration::hasValidSignature($uri)) {
                            $fail(Lang::get('registration.create.user.validation.failed'));
                        }
                    },
                ],
                'expires' => 'required|numeric',
            ]
        );

        // Let's create our user !
        $bridge = new BridgeContent();

        $data = [
            'email' => $request->get('email'),
            'password' => $request->get('password'),
            'name' => $request->get('email'),
            'is_maintenance' => 0,
            'status' => 1,
            'is_from_ldap' => 0,
            'langcode' => 'fr',
            'registration_status' => 'pending',
        ];

        $user = Users::query()->where('email', $request->get('email'))->first();

        if ($user) {
            $data['uuid'] = $user->uuid;
        }

        $uuid = $bridge->contentInsert('users', $data);

        // Check user is OK
        $insideUser = Users::where('email', $request->get('email'))->first();
        if ($insideUser && ($insideUser->uuid == $uuid)) {
            return new JsonResponse(['status' => true]); // return 200 means ok for our front
        }

        return new JsonResponse(['error' => Lang::get('registration.create.user.failed')], 400);
    }


    /**
     * Permet de valider la signature
     *
     * @param Request $request
     * @return array
     */
    public function validateRegistration(Request $request): array
    {
        Lang::setLocale('fr');

        $this->validate(
            $request,
            [
                'email' => 'required|email',
                'signature' => [
                    'required',
                    'string',
                    function ($attribute, $value, $fail) use ($request) {
                        $uri = Registration::buildUrl(parse_url(config('app.url') . '/start-registration'), [
                            'email' => str_replace(' ', '+', $request->get('email')),
                            'expires' => $request->get('expires'),
                            'signature' => $value,
                        ]);
                        if (!Registration::hasValidSignature($uri)) {
                            $fail(Lang::get('registration.create.user.validation.failed'));
                        }
                    },
                ],
                'expires' => 'required|numeric',
            ]
        );

        return ['status' => true]; // return 200 means ok for our front
    }

    /**
     * Complete the registration and send mail to admin
     *
     * @param Request $request
     * @param string $uuid
     * @return bool[]|\Illuminate\Http\JsonResponse
     */
    public function completeRegistration(Request $request, string $uuid)
    {
        if (!$uuid) {
            return new JsonResponse(['error' => Lang::get('registration.create.user.failed')], 400);
        }

        $user = Users::query()->find($uuid);

        if (!$user) {
            return new JsonResponse(['error' => Lang::get('registration.create.user.failed')], 400);
        }

        $bridge = new BridgeContent();

        $data = array_merge($request->all(), ['uuid' => $uuid]);

        $bridge->contentUpdate('users', $data);

        Mail::to(env('TRYB_REGISTRATION_ADMIN_MAIL', 'dpfister@tryba.fr'))->send(new PendingAccount($user->uuid, $request->all()));

        return ['status' => true];
    }

    /**
     * Update user registration status
     *
     * @param Request $request
     * @param string $uuid
     * @return bool[]|\Illuminate\Http\JsonResponse
     */
    public function updateUserStatus(Request $request, string $uuid)
    {
        $user = Users::query()->find($uuid);

        if (!$user) {
            return new JsonResponse(['error' => Lang::get('registration.user.404')], 400);
        }

        $message = $request->get('registration_message');
        $status = $request->get('registration_status');

        $bridge = new BridgeContent();

        $bridge->contentUpdate('users', [
            'uuid' => $uuid,
            'registration_status' => $status,
            'registration_message' => $message
        ]);

        $message = empty($message) ? '' : $message;
        Mail::to($user->email)->send(new AccountStatusChanged($user->uuid, $status, $message));

        return ['status' => true];
    }
}
