<?php

use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Entities;
use Inside\Content\Models\Contents\Services;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;

return [
    'sync_attributes' => [
        'mail' => 'mail',
        'name' => 'displayName',
        'lastname' => 'surname',
        'firstname' => 'givenName',
        'function' => 'jobTitle',
        'mobile_phone' => 'mobilePhone',
        'phone' => 'businessPhones',
        'services' => 'department',
        'manager' => static function ($user, BridgeContent $bridge): ?string {
            if (! isset($user->manager)) {
                return null;
            }

            return Users::select('uuid', 'email')->where('email', $user->manager)->where('status', 1)->first()?->uuid;
        },
        'status' => static fn (): bool => true,
        'is_maintenance' => static fn (): bool =>  false,
    ],
    'postfilter' => static function (Collection $users): Collection {
        $bridge = new BridgeContent();

        $users = $users->filter(fn (mixed $user) => $user->accountEnabled);

        $mainDomain = collect($users->pluck('mail'))
            ->filter()
            ->map(fn ($email) => (string) str($email)->lower()->after('@'))
            ->countBy()
            ->sort()
            ->keys()
            ->last();

        $provider = match ($mainDomain) {
            'touraine-logement.fr' => 'touraine_logement',
            'meduane-habitat.fr' => 'meduane_habitat',
            'mancelle-habitation.fr' => 'mancelle_habitation',
            default => 'lb_habitat'
        };

        $alreadyImportedUsers = User::all()->pluck('provider_name', 'email');

        $servicesToImport = $users
            ->pluck('department')
            ->filter()
            ->map(fn ($service) => (string) str($service)->lower()->ucfirst())
            ->unique();

        $services = Services::all()->pluck('uuid', 'title');

        $servicesToImport
            ->diff($services->keys())
            ->each(fn ($service) => $services[$service] = $bridge->contentInsert('services', [
                'title' => $service,
                'status' => 1,
                'author' => Auth::user()?->uuid,
                'langcode' => 'fr',
                'published_at' => Carbon::now(),
            ]));

        return $users
            ->filter(fn ($user) => $user->givenName && $user->surname && $user->mail && $user->companyName)
            ->map(function (stdClass $user) use ($services) {
                $user->businessPhones = Arr::first($user->businessPhones);
                $user->department = $services->get((string) str($user->department ?? '')->lower()->ucfirst(), null);
                $user->givenName = (string) str($user->givenName)->lower()->title();
                $user->surname = (string) str($user->surname)->lower()->title();
                $user->mail = (string) str($user->mail)->lower();

                return $user;
            })
            ->reject(fn ($user) => $alreadyImportedUsers->has($user->mail) && $alreadyImportedUsers->get($user->mail) !== $provider);
    },
    'postflight' => static function ($console) {
        $bridge = new BridgeContent();

        $entities = Entities::query()
            ->get(['uuid', 'provider_name'])
            ->pluck('uuid', 'provider_name');

        $azureadUsersUuid = User::query()
            ->where('provider_type', 'azuread')
            ->get(['uuid', 'provider_name'])
            ->pluck('provider_name', 'uuid')
            ->transform(fn ($entity) => $entities->get($entity));

        $users = Users::query()->whereDoesntHave('entities')->whereIn('uuid', $azureadUsersUuid->keys()->toArray())->get(['uuid']);

        foreach ($users as $user) {
            $bridge->contentUpdate('users', [
                'type' => 'users',
                'uuid' => $user->uuid,
                'entities' => $azureadUsersUuid->get($user->uuid),
            ]);
        }
    },
    'custom_attributes' => [
        'manager' => 'mail',
    ],
    'azuread_domains' => [
        'touraine_logement' => [
            'app_id' => env('TOURAINE_AZUREAD_APP_ID'),
            'secret_key' => env('TOURAINE_AZUREAD_SECRET_KEY'),
            'app_tenant' => env('TOURAINE_AZUREAD_TENANT'),
        ],
        'meduane_habitat' => [
            'app_id' => env('MEDUANE_AZUREAD_APP_ID'),
            'secret_key' => env('MEDUANE_AZUREAD_SECRET_KEY'),
            'app_tenant' => env('MEDUANE_AZUREAD_TENANT'),
        ],
        'mancelle_habitation' => [
            'app_id' => env('MANCELLE_AZUREAD_APP_ID'),
            'secret_key' => env('MANCELLE_AZUREAD_SECRET_KEY'),
            'app_tenant' => env('MANCELLE_AZUREAD_TENANT'),
        ],
        'lb_habitat' => [
            'app_id' => env('LB_AZUREAD_APP_ID'),
            'secret_key' => env('LB_AZUREAD_SECRET_KEY'),
            'app_tenant' => env('LB_AZUREAD_TENANT'),
        ],
    ],
];
