<?php

namespace Inside\SYAN\Services;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Str;
use Inside\Host\Bridge\BridgeContent;
use League\Csv\ByteSequence;
use League\Csv\CharsetConverter;
use League\Csv\Exception;
use League\Csv\Reader;

class ImportCsvService
{
    /**
     * @var BridgeContent
     */
    private $bridge;

    /**
     * @param BridgeContent $bridge
     */
    public function __construct(BridgeContent $bridge)
    {
        $this->bridge = $bridge;
    }

    /**
     * We take the csv file and put it in an array with the right keys and values
     *
     * @param string $filePath
     * @param string $delimiter
     *
     * @return array
     * @throws Exception
     */
    public function getCsvRecords(string $filePath, string $delimiter): array
    {
        $csv = Reader::createFromPath($filePath);
        $encoding = mb_detect_encoding($csv->toString(), mb_list_encodings(), true);

        if ($encoding !== 'UTF-8') {
            $csv->setOutputBOM(ByteSequence::BOM_UTF8);
            $csv->addStreamFilter(CharsetConverter::getFiltername((string) $encoding, 'UTF-8'));
        }

        $csv->setDelimiter($delimiter);
        $csv->setHeaderOffset(0);

        $headers = array_map(
            function ($item) {
                return Str::lower(trim($item));
            },
            $csv->getHeader()
        );

        return iterator_to_array($csv->getRecords($headers));
    }

    /**
     * @param array $record
     * @param string $bundle
     * @param string $column
     * @param string $separator
     *
     * @return array|null
     * @throws \Exception
     */
    public function getEntityReference(array $record, string $bundle, string $column, string $separator = ''): ?array
    {
        $separator = empty($separator) ? ',' : $separator;
        $query = call_user_func(type_to_class($bundle).'::get');
        if (Str::contains($record[$column], $separator)) {
            $column = explode($separator, $record[$column]);
        }
        $dataUuid = [];
        if (is_array($column)) {
            foreach ($column as $name) {
                $name = Str::ucfirst(trim($name));
                if (empty($name)) {
                    continue;
                }
                $dataUuid = $this->getUuidReference($dataUuid, $query, $name, $bundle);
            }

            return $dataUuid;
        } else {
            $name = Str::ucfirst(trim($record[$column]));
            if (empty($name)) {
                return null;
            }

            return $this->getUuidReference($dataUuid, $query, $name, $bundle);
        }
    }

    /**
     * @param array $dataUuid
     * @param Collection $query
     * @param string $name
     * @param string $bundle
     *
     * @return mixed
     * @throws \Exception
     */
    protected function getUuidReference(array $dataUuid, Collection $query, string $name, string $bundle)
    {
        $entity = $query->where('title', $name)->first();
        if ($entity && isset($entity->uuid)) {
            $dataUuid[] = $entity->uuid;
        } else {
            $dataUuid[] = $this->contentInsertReference($name, $bundle);
        }

        return $dataUuid;
    }

    /**
     * Builds the array and inserts the content type in DB
     *
     * @param string $name
     * @param string $bundle
     *
     * @return string|null
     * @throws \Exception
     */
    protected function contentInsertReference(string $name, string $bundle): ?string
    {
        return $this->bridge->contentInsert($bundle, [
            'title' => $name,
            'type' => 'node',
            'bundle' => $bundle,
        ]);
    }
}
