<?php

namespace Inside\SYAN\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\File;
use Inside\Content\Models\Contents\Events;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\SYAN\Facades\ImportCsv;
use Inside\SYAN\Services\Horoquartz\HoroquartzRequest;
use Inside\SYAN\Services\Horoquartz\HoroquartzService;
use Inside\User\Models\User;
use Symfony\Component\Console\Helper\ProgressBar;

class HoroquartzStatsForMatricule extends Command
{
    /**
     * @var string
     */
    protected $name = 'syan:horoquartz:stats:matricule';

    /**
     * @var string
     */
    protected $signature = 'syan:horoquartz:stats:matricule {matricule}';

    /**
     * @var string
     */
    protected $description = 'Get stats around horoquartz for a specific matricule';

    public function handle(): void
    {
        $matricule = $this->argument('matricule');

        if (blank($matricule) || ! is_string($matricule)) {
            $this->error('Matricule is required');

            return;
        }

        $this->line('');
        $this->info('SOAP Request - Solde de congés');
        $vacationBalanceRequest = HoroquartzRequest::envelopeSalesForce(
            service: 'getEmployeeCounters',
            request: 'waxRequest',
            params: [
                'codeFct' => 'WB1',
                'matricule' => $matricule,
            ]
        );

        $request = HoroquartzRequest::post(config('horoquartz.url.services'), $vacationBalanceRequest);
        dump($request->response());

        $this->line('');
        $this->info('SOAP Request - Cumul de la veille');
        $culatedDayBeforeRequest = HoroquartzRequest::envelopeSalesForce(
            service: 'getEmployeeCounters',
            request: 'waxRequest',
            params: [
                'codeFct' => 'WA10',
                'matricule' => $matricule,
            ]
        );

        $request = HoroquartzRequest::post(config('horoquartz.url.services'), $culatedDayBeforeRequest);
        dump($request->response());

        $this->line('');
        $this->info('SOAP Request - Pointage du jour');
        $dailyBadgingRequest = HoroquartzRequest::envelopeSalesForce(
            service: 'clocksOfTheDay',
            request: 'wclocksRequest',
            params: [
                'date' => Carbon::now()->timezone('Europe/Paris')->format('Y-m-d'),
                'matricule' => $matricule,
            ]
        );

        $request = HoroquartzRequest::post(config('horoquartz.url.services'), $dailyBadgingRequest);
        dump($request->response());

        $horoquartz = new HoroquartzService($matricule);
        $vacationBalance = $horoquartz->getVacationBalance();
        $cumulatedDayBefore = $horoquartz->getCumulatedDayBefore();
        $dailyBadging = $horoquartz->getDailyBadging();
        $direction = match ($dailyBadging['direction']) {
            '0' => 'Entrée',
            '1' => 'Sortie',
            default => 'N\A',
        };
        $hour = match (true) {
            ! blank($dailyBadging['hour']) => Date::createFromTimestamp($dailyBadging['hour'])
                ->timezone('Europe/Paris')
                ->format('H:i'),
            default => 'N\A',
        };

        $this->line('');
        $this->info('Statistique Utilisateurs');
        $this->table(['Matricule', 'Solde de congés', 'Cumul de la veille', 'Pointage du jour'], [
            [$matricule, $vacationBalance, $cumulatedDayBefore, $direction.' à '.$hour],
        ]);
    }
}
