<?php

namespace Inside\SYAN\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\File;
use Inside\Content\Models\Contents\Events;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\SYAN\Facades\ImportCsv;
use Inside\SYAN\Services\Horoquartz\HoroquartzService;
use Inside\User\Models\User;
use Symfony\Component\Console\Helper\ProgressBar;

class HoroquartzStats extends Command
{
    /**
     * @var string
     */
    protected $name = 'syan:horoquartz:stats';

    /**
     * @var string
     */
    protected $signature = 'syan:horoquartz:stats';

    /**
     * @var string
     */
    protected $description = 'Get stats around horoquartz';

    public function handle(): void
    {
        $uuids = User::where('provider_type', 'azuread')->where('status', true)->pluck('uuid');
        $users = Users::query()->whereIn('uuid', $uuids)->get();

        $this->info('Statistique Utilisateurs');
        $this->comment(' - Utilisateur sans matricule: '.$withoutMatricule = $users->whereNull('matricule')->count());
        $this->comment(' - Utilisateur avec un matricule erroné: '.$wrongMatricule = $users->whereNotNull('matricule')->reject(fn ($user) => ctype_digit($user->matricule))->count());
        $this->comment(' - Utilisateur avec un matricule valide: '.$users->whereNotNull('matricule')->filter(fn ($user) => ctype_digit($user->matricule))->count());

        if ($withoutMatricule > 0) {
            $this->line('');
            $this->info('Utilisateur sans matricule:');
            $this->table(['Nom', 'Prénom', 'Email'], $users->whereNull('matricule')->map(fn ($user) => [$user->lastname, $user->firstname, $user->email]));
        }

        if ($wrongMatricule > 0) {
            $this->line('');
            $this->info('Utilisateur avec un matricule erroné:');
            $this->table(['Nom', 'Prénom', 'Email', 'Matricule'], $users->whereNotNull('matricule')->reject(fn ($user) => ctype_digit($user->matricule))->map(fn ($user) => [$user->lastname, $user->firstname, $user->email, $user->matricule]));
        }

        $users = $users
            ->whereNotNull('matricule')
            ->filter(fn ($user) => ctype_digit($user->matricule))
            ->map(function (Users $user) {
                $horoquartz = new HoroquartzService($user->matricule);
                $vacationBalance = $horoquartz->getVacationBalance();
                $cumulatedDayBefore = $horoquartz->getCumulatedDayBefore();
                $dailyBadging = $horoquartz->getDailyBadging();
                $direction = match ($dailyBadging['direction']) {
                    '0' => 'Entrée',
                    '1' => 'Sortie',
                    default => 'N\A',
                };
                $hour = match (true) {
                    ! blank($dailyBadging['hour']) => Date::createFromTimestamp($dailyBadging['hour'])
                        ->timezone('Europe/Paris')
                        ->format('H:i'),
                    default => 'N\A',
                };

                return [
                    'Nom' => $user->lastname,
                    'Prénom' => $user->firstname,
                    'Email' => $user->email,
                    'Matricule' => $user->matricule,
                    'Solde congés' => $vacationBalance,
                    'Congés à venir' => $cumulatedDayBefore,
                    'Dernier badgeage' => $direction.' à '.$hour,
                ];
            });

        $this->line('');
        $this->info('Statistique Horoquartz');
        $this->table(['Nom', 'Prénom', 'Email', 'Matricule', 'Solde congés', 'Congés à venir', 'Dernier badgeage'], $users);
    }
}
