<?php

namespace Inside\SYAN\Services\Horoquartz;

use Illuminate\Support\Facades\Cache;
use Inside\SYAN\Services\Horoquartz\Exceptions\HoroquartzException;
use Inside\SYAN\Services\Horoquartz\Exceptions\NotAuthorizedIP;
use Inside\SYAN\Services\Horoquartz\Exceptions\UnableToReachHoroquartz;

class HoroquartzToken
{
    /**
     * @throws HoroquartzException
     * @throws NotAuthorizedIP
     * @throws UnableToReachHoroquartz
     */
    public function __construct()
    {
        if ($this->valid()) {
            return;
        }

        $request = $this->generate();

        $request->throw();

        $this->store($request->response());
    }

    /**
     * @throws HoroquartzException
     * @throws NotAuthorizedIP
     * @throws UnableToReachHoroquartz
     */
    public function valid(): bool
    {
        if (! Cache::has('token-horoquartz') || ! Cache::has('application-horoquartz')) {
            return false;
        }

        $envelope = HoroquartzRequest::envelopeAuthenticate(
            service: 'VerifyToken',
            request: 'verificationRequest',
            params: [
                'TokenID' => Cache::get('token-horoquartz'),
                'ApplicationID' => Cache::get('application-horoquartz'),
            ]
        );

        $request = HoroquartzRequest::post(config('horoquartz.url.authentication'), $envelope);

        $request->throw();

        return $request->response()->contains('Application access granted');
    }

    public function generate(): HoroquartzRequest
    {
        $envelope = HoroquartzRequest::envelopeAuthenticate(
            service: 'Authenticate',
            request: 'authenticationRequest',
            params: [
                'User' => config('horoquartz.user'),
                'Password' => config('horoquartz.password'),
            ]
        );

        return HoroquartzRequest::post(config('horoquartz.url.authentication'), $envelope);
    }

    public function store(HoroquartzResponse $response): void
    {
        Cache::put(
            key: 'token-horoquartz',
            value: $response->searchBetween('<Token><ID>', '</ID>'),
            ttl: 7200
        );

        Cache::put(
            key: 'application-horoquartz',
            value: $response->searchBetween('<Applications><ID>', '</ID>'),
            ttl: 7200
        );
    }
}
