<?php

namespace Inside\SYAN\Services\Horoquartz;

use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Cache;
use Inside\SYAN\Services\Horoquartz\Exceptions\HoroquartzException;
use Inside\SYAN\Services\Horoquartz\Exceptions\NotAuthorizedIP;
use Inside\SYAN\Services\Horoquartz\Exceptions\UnableToReachHoroquartz;
use Psr\Http\Message\ResponseInterface;

class HoroquartzRequest
{
    protected GuzzleClient $client;

    protected ?HoroquartzException $exception = null;

    protected ?ResponseInterface $response = null;

    private function __construct(string $method, string $uri, string $body)
    {
        try {
            $client = new GuzzleClient();

            $this->response = $client->request(
                method: $method,
                uri: $uri,
                options: [
                    'headers' => [
                        'Content-type' => 'text/xml;charset="utf-8"',
                        'Cache-Control' => 'no-cache',
                        'Pragma' => 'no-cache',
                        'Content-length' => strlen($body),
                    ],
                    'verify' => true,
                    'timeout' => 10,
                    'body' => $body,
                ]
            );
        } catch (GuzzleException $exception) {
            $this->exception = match ($exception->getCode()) {
                403 => NotAuthorizedIP::create(),
                default => UnableToReachHoroquartz::create(),
            };
        }
    }

    public function successful(): bool
    {
        return is_null($this->exception);
    }

    /**
     * @throws HoroquartzException
     * @throws NotAuthorizedIP
     * @throws UnableToReachHoroquartz
     */
    public function throw(): void
    {
        if (is_null($this->exception)) {
            return;
        }

        throw $this->exception::create();
    }

    public function response(): HoroquartzResponse
    {
        return new HoroquartzResponse($this->response);
    }

    public static function post(string $uri, string $body): self
    {
        return new self('post', $uri, $body);
    }

    public static function envelopeAuthenticate(string $service, string $request, array $params = []): string
    {
        return view(
            view: 'soap.auth-envelope',
            data: compact('service', 'request', 'params')
        )->toHtml();
    }

    public static function envelopeSalesForce(string $service, string $request, array $params = []): string
    {
        $token = Cache::get('token-horoquartz');

        return view(
            view: 'soap.salesforce-envelope',
            data: compact('token', 'service', 'request', 'params')
        )->toHtml();
    }
}
