<?php

namespace Inside\SYAN\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\File;
use Inside\Content\Models\Contents\Events;
use Inside\Host\Bridge\BridgeContent;
use Inside\SYAN\Facades\ImportCsv;
use Symfony\Component\Console\Helper\ProgressBar;

class ImportCsvCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'syan:import';

    /**
     * @var string
     */
    protected $signature = 'syan:import {type}
                                        {--d|delimiter=}
                                        {--p|path=}';

    /**
     * @var string
     */
    protected $description = 'Import contents';

    private ?BridgeContent $bridge = null;

    public function handle(): void
    {
        /** @var string $type */
        $type = $this->argument('type');

        /** @var string $delimiter */
        $delimiter = empty($this->option('delimiter')) ? ';' : $this->option('delimiter');

        /** @var string $filePath */
        $filePath = empty($this->option('path')) ? __DIR__.'/../../../../../storage/app/csv/'.$type.'.csv' : $this->option('path');

        if (! File::exists($filePath)) {
            $this->error('File not found');

            return;
        }

        $this->writeln("<comment>Début de l'import $type !</comment>");
        $records = ImportCsv::getCsvRecords($filePath, $delimiter);
        $this->writeln(sprintf('<info>Il y a [%d] catalogues %s à importer dans le csv %s</info>', count($records), $type, pathinfo($filePath)['basename']));

        $this->buildContent($records, $type);
    }

    protected function buildContent(array $records, string $type): void
    {
        $value = config($type.'_import'.'.'.$type);

        $bar = $this->getOutput()->createProgressBar(count($records));

        $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");

        $identifier = 'title';

        if ($type === 'events') {
            $identifier = 'external_id';
        }

        $imported = [];
        foreach ($records as $record) {
            $data = [];

            foreach ($value as $modelField => $fileField) {
                if (($fileField instanceof \Closure)) {
                    $data[$modelField] = $fileField($record);
                } else {
                    $data[$modelField] = $record[$fileField];
                }
            }

            if (empty($data[$identifier])) {
                continue;
            }

            $imported[] = $this->insertContent($data, $bar, $identifier);
        }

        $bar->finish();
        $this->getOutput()->newLine();

        if ($type === 'events') {
            $this->deleteNotImportedFutureExternalEvents($imported);
        }
    }

    protected function insertContent(array $data, ProgressBar $bar, string $identifier): ?string
    {
        $query = call_user_func(type_to_class($data['bundle']).'::query');
        $existingUuid = $query->where($identifier, $data[$identifier])->first()?->uuid;

        if ($existingUuid !== null) {
            $data['uuid'] = $existingUuid;
        }

        $bar->setMessage("Importation du contenu => {$data['title']}");
        $uuid = $this->getBridge()->contentInsert($data['bundle'], $data, ! isset($data['uuid']));
        $bar->advance();

        return $uuid;
    }

    protected function writeln(string $message): void
    {
        $this->getOutput()->writeln($message);
    }

    private function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    private function deleteNotImportedFutureExternalEvents(array $imported): void
    {
        $events = Events::query()
            ->whereNotNull('external_id')
            ->whereNotIn('uuid', $imported)
            ->whereDate('date', '>', Date::now()->toDateString())
            ->get();

        if ($events->isEmpty()) {
            $this->writeln('Aucun évènement à supprimer');

            return;
        }

        $this->writeln('Suppression des futurs évènements non importés');

        $bar = $this->getOutput()->createProgressBar($events->count());
        $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");

        $events->each(function (Events $event) use ($bar) {
            $this->getBridge()->contentDelete('events', $event->uuid);
            $bar->setMessage("Suppression du contenu => $event->title");
            $bar->advance();
        });

        $bar->finish();
        $this->getOutput()->newLine();
        $this->writeln('Suppression terminée');
    }
}
