<?php

namespace Inside\SOFI\Services;

use DateTime;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Contracts\WysiwygImageService;
use Inside\Support\Str;

class TikehauService
{
    public function __construct(
        private WysiwygImageService $wysiwygImageService
    ) {
    }

    /**
     * @param string $url
     * @return bool|string
     */
    public function getTikehauResource(string $url)
    {
        $ch = curl_init();

        curl_setopt($ch, CURLOPT_HTTPGET, 1);

        if (config('tikehau.serviceAccount')) {
            curl_setopt($ch, CURLOPT_GSSAPI_DELEGATION, CURLGSSAPI_DELEGATION_FLAG);
            curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_GSSNEGOTIATE);
            curl_setopt($ch, CURLOPT_USERPWD, config('tikehau.serviceAccount'));
        }

        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        $resultContent = curl_exec($ch);

        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if (curl_errno($ch)) {
            return false;
            //            $msg = curl_error($ch);
            //            curl_close($ch);
            //            die('Curl error: content: '.$resultContent.' code: '.$httpCode);
        }

        curl_close($ch);

        return $resultContent;
    }

    /**
     * @param string $fileUrl
     * @return string
     * @throws \Exception
     */
    public function uploadFileAndImage(string $fileUrl): string
    {
        $randomInt = random_int(1000000, 9000000000);
        $nameFile = class_basename($fileUrl);
        $filePath = 'chunks/'.$randomInt.'/'.$nameFile;
        $download = $this->getTikehauResource($fileUrl);

        //means there is an error, else it's a string
        if (is_bool($download)) {
            throw new \Exception("error while retrieving Tikehau resource");
        }

        Storage::disk('local')->put($filePath, $download);

        return $filePath;
    }

    /**
     * @param mixed $responseData
     * @return array
     * @throws \Exception
     */
    public function manageParagraphForData($responseData): array
    {
        $contents['content'] = [];
        if ($responseData->body->value !== null) {
            $contents['content'][] = ['bundle' => 'text', 'body' => $this->uploadBodyImages($responseData->body->value)];
        }

        if ($responseData->files !== null) {
            foreach ($responseData->files as $file) {
                $filePath = $this->uploadFileAndImage($file->url);
                $contents['content'][] = ['bundle' => 'file', 'file' => $filePath];
            }
        }

        return $contents;
    }

    public function uploadBodyImages(string $body): string
    {
        $images = [];

        if (preg_match_all('#<img .*\ssrc="(/[^"]+)".*/>#', $body, $images)) {
            [$tags, $relativePaths] = $images;

            foreach ($relativePaths as $relativePath) {
                $imageUrl = $this->getTikehauUrl($relativePath);
                $wwImage = $this->uploadWysiwygImage($imageUrl);
                $body = Str::replace($relativePath, $wwImage, $body);
            }
        }

        return $body;
    }

    public function getTikehauUrl(string $endpoint): string
    {
        $apiUrl = rtrim(config('tikehau.apiUrl'), '/');
        $endpoint = ltrim($endpoint, '/');

        return "$apiUrl/$endpoint";
    }

    public function uploadWysiwygImage(string $imageUrl): string
    {
        $downloadedPath = $this->uploadFileAndImage($imageUrl);
        $uploadedFile = new UploadedFile(
            Storage::disk('local')->path($downloadedPath), // @phpstan-ignore-line
            basename($downloadedPath),
        );

        return url($this->wysiwygImageService->upload($uploadedFile));
    }

    /**
     * @param string $contentType
     * @return false|int
     * @throws \Exception
     */
    public function getLastTikehauDate(string $contentType)
    {
        $newsTikehau = DB::table('inside_content_'.$contentType)
            ->orderByDesc('created_at')
            ->first();

        if (! $newsTikehau) {
            return false;
        }

        $dateCreated = new DateTime($newsTikehau->created_at);
        $dateCreated = $dateCreated->getTimestamp();

        return $dateCreated + 1;
    }
}
