<?php

use Adldap\Models\Entry;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;

if (!function_exists('get_trim_attribute')) {
    function get_trim_attribute(Entry $user, string $attributeName): ?string
    {
        $attribute = trim($user->getFirstAttribute($attributeName));
        if (empty($attribute)) { // because trim(null) = ""
            return null;
        }

        return $attribute;
    }
}

function getEntityReference(Entry $user, BridgeContent $bride, string $attribute, string $contentType): ?array
{
    if (!$user->hasAttribute($attribute)) {
        return null;
    }

    $uuids = [];
    $titles = $user->getAttribute($attribute);

    foreach ($titles as $title) {
        $entity = call_user_func(type_to_class($contentType).'::query')
            ->where('title', $title)
            ->first();

        if ($entity) {
            $uuid = $entity->uuid;
        } else {
            $uuid = $bride->contentInsert(
                $contentType,
                [
                    'type' => 'node',
                    'bundle' => $contentType,
                    'title' => $title,
                    'uid' => 1,
                ]
            );
        }

        $uuids[] = $uuid;
    }

    return empty($uuids) ? null : $uuids;
}

return [
    'user_filter' => 'is_from_ldap',
    'connection' => env('ADLDAP_CONNECTION', 'default'),
    'provider' => Adldap\Laravel\Auth\DatabaseUserProvider::class,
    'rules' => [
        Adldap\Laravel\Validation\Rules\DenyTrashed::class,
    ],
    'sync_attributes' => [
        'mail'          => 'mail',
        'name'          => 'samaccountname',
        'firstname'     => 'givenname',
        'lastname'      => 'sn',
        'mobile'        => 'mobile',
        'phone'         => 'telephonenumber',
        'direct_line'   => 'othertelephone',
        'function'      => function (Entry $user, BridgeContent $bridge) {
            return get_trim_attribute($user, 'title');
        },
        'services'       => function (Entry $user, BridgeContent $bridge) {
            $company = str(get_trim_attribute($user, 'company') ?? '')->trim()->lower();
            if (in_array($company, ['tc', 'tim'])) {
                return [];
            }

            return getEntityReference($user, $bridge, 'department', 'services');
        },
        'locations'     => function (Entry $user, BridgeContent $bridge) {
            $company = str(get_trim_attribute($user, 'company') ?? '')->trim()->lower();
            if (in_array($company, ['tc', 'tim'])) {
                return [];
            }

            return getEntityReference($user, $bridge, 'L', 'locations');
        },
        'entities'     => function (Entry $user, BridgeContent $bridge) {
            return getEntityReference($user, $bridge, 'company', 'entities');
        },
        'status'        => function (Entry $user, BridgeContent $bridge) {
            return 1;
        },
        'image' => function (Entry $user, BridgeContent $bridge) {
            $fileName = strtolower(trim($user->getFirstAttribute('givenname')) . '.' . trim($user->getFirstAttribute('sn')));
            $fileName = str_replace([' ', '\''], ['-', ''], $fileName);
            $fileName = strtr(utf8_decode($fileName), utf8_decode('àáâãäçèéêëìíîïñòóôõöùúûüýÿÀÁÂÃÄÇÈÉÊËÌÍÎÏÑÒÓÔÕÖÙÚÛÜÝ'), 'aaaaaceeeeiiiinooooouuuuyyAAAAACEEEEIIIINOOOOOUUUUY');
            $imgPath = "profile/$fileName";
            $insideUser = Users::query()->where('email', $user->getFirstAttribute('mail'))->first();

            if (Storage::exists("$imgPath.jpg")) {
                $imgPath .= '.jpg';
            } elseif (Storage::exists("$imgPath.jfif")) {
                $imgPath .= '.jfif';
            } else {
                return $insideUser ? $insideUser->image : null;
            }

            if ($insideUser && $insideUser->image) {
                return $insideUser->image;
            }

            $chunkId = Str::random(32);
            $finalFolderPath = "chunks/$chunkId";
            $finalPath = "$finalFolderPath/$fileName";

            Storage::makeDirectory($finalFolderPath);
            Storage::copy($imgPath, $finalPath);

            return $finalPath;
        }
    ],
];
