<?php

namespace Inside\SOFI\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\Contents\Jobs;
use Inside\Host\Bridge\BridgeContent;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use League\Csv\Writer;

class ExportJobsContentsCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'sofi:export-jobs';

    /**
     * @var string
     */
    protected $signature = 'sofi:export-jobs';

    /**
     * @var string
     */
    protected $description = 'Export jobs contents to csv file';

    /**
     * @var Jobs
     */
    private $job;

    public function handle(): void
    {
        $header = ['title', 'services', 'contracts', 'locations', 'main_location', 'published_at', 'email', 'description'];

        $csvFile = Writer::createFromString();
        $csvFile->setOutputBOM(Writer::BOM_UTF8);
        $csvFile->insertOne($header);

        $jobsContents = Jobs::all();

        /** @var Jobs $job */
        foreach ($jobsContents as $job) {
            $this->job = $job;

            $jobValues = [
                'title'         => $job->getAttributeValue('title'),
                'services'      => $this->getRelationFieldValues('services'),
                'contracts'     => $this->getRelationFieldValues('contracts'),
                'locations'     => $this->getRelationFieldValues('locations'),
                'main_location' => $this->getRelationFieldValues('mainLocation'),
                'published_at'    => $job->getAttributeValue('published_at') ? Carbon::createFromTimestampUTC($job->getAttributeValue('published_at'))->format('Y/m/d') : '',
                'email'         => trim($job->getAttributeValue('email')),
                'description'   => $this->getJobDescriptionValue(),
            ];

            $csvFile->insertOne($jobValues);
        }

        Storage::disk('protected')->put('export/job_offers.csv', $csvFile->getContent());
    }

    /**
     * @param string $jobsFieldName
     * @return string
     */
    private function getRelationFieldValues(string $jobsFieldName): string
    {
        $values = [];
        $relations = $this->job->getAttribute($jobsFieldName);
        if (!$relations) {
            return '';
        }

        foreach ($relations as $relation) {
            $values[] = $relation->getAttributeValue('title');
        }

        return implode(',', $values);
    }

    /**
     * @return string
     */
    private function getJobDescriptionValue(): string
    {
        try {
            $section = $this->job->section_content[0];
            return $section ? $section->getAttributeValue('body') : '';
        } catch (\Exception $exception) {
            return '';
        }
    }
}
