<?php

namespace Inside\SETE\Services;

use RuntimeException;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Inside\Facades\Http;
use Inside\SETE\Exceptions\SheetDownloadException;
use Inside\Support\Http\Client\Response;
use Laravel\Lumen\Http\ResponseFactory;
use Symfony\Component\HttpFoundation\StreamedResponse;

class EiffelApiService
{
    private string $eiffelCertPath;
    private string $eiffelKeyPath;

    public function __construct(
        private string $eiffelApiBaseUri,
        private string $eiffelApiUser,
        private string $eiffelApiMatricule,
        private string $eiffelApiPassword,
        private string $eiffelCertPem,
        private string $eiffelKeyPem,
        private string $eiffelCertPwd
    ) {
        Log::info('[Eiffel API Service] : Initializing EiffelApiService');

        $eiffelCertPath = @tempnam(sys_get_temp_dir(), 'cert_');
        if (false === $eiffelCertPath) {
            throw new RuntimeException('Impossible de créer un fichier temporaire pour le certificat.');
        }

        if (false === file_put_contents($eiffelCertPath, $eiffelCertPem, LOCK_EX)) {
            @unlink($eiffelCertPath);
            throw new RuntimeException('Impossible d’écrire le fichier cert.');
        }
        Log::info('[Eiffel API Service] : Cert file created at ' . $eiffelCertPath);

        $eiffelKeyPath  = @tempnam(sys_get_temp_dir(), 'key_');
        if (false === $eiffelKeyPath) {
            @unlink($eiffelCertPath);
            throw new RuntimeException('Impossible de créer un fichier temporaire pour la clé.');
        }

        if (false === file_put_contents($eiffelKeyPath, $eiffelKeyPem, LOCK_EX)) {
            @unlink($eiffelKeyPath);
            throw new RuntimeException('Impossible d’écrire le fichier key.');
        }
        Log::info('[Eiffel API Service] : Key file created at ' . $eiffelKeyPath);

        $this->eiffelCertPath = $eiffelCertPath;
        $this->eiffelKeyPath  = $eiffelKeyPath;

        register_shutdown_function(function () use ($eiffelCertPath, $eiffelKeyPath) {
            if (file_exists($eiffelCertPath)) {
                @unlink($eiffelCertPath);
            }
            if (file_exists($eiffelKeyPath)) {
                @unlink($eiffelKeyPath);
            }
        });
    }

    private array $extraParameters = [];

    public function addExtraParameter(string $key, ?string $value): self
    {
        $this->extraParameters[$key] = $value;

        return $this;
    }

    public function downloadFile(string $action, string $date): StreamedResponse
    {
        /**
         * @var ResponseFactory $response
         */
        $response = response();
        try {
            $apiResponse = $this->getResponse($action, $date);
            if ($apiResponse->failed()) {
                throw new SheetDownloadException('Failed to download the file from API');
            }

            $bodyDecoded = json_decode($apiResponse->body(), true);
            if (!empty($bodyDecoded['erreur'])) {
                throw new SheetDownloadException("Failed to download the file from API with message {$bodyDecoded['erreur']}");
            }

            $filename = "{$action}-" . $date . ".pdf";

            return $response->stream(function () use ($apiResponse) {
                echo $apiResponse->body();
            }, 200, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => "inline; filename=\"{$filename}\"",
            ]);
        } catch (\Exception $exception) {
            Log::warning('[Eiffel API Service] : Error HTTP Response', [
                'message' => $exception->getMessage(),
            ]);

            return $response->stream(function () {
                echo 'Error downloading file';
            }, 500, [
                'Content-Type' => 'text/plain',
            ]);
        }
    }

    public function getAction(string $action, ?string $date = null): JsonResponse
    {
        try {
            $response = $this->getResponse($action, $date);
            $body = $response->body();
            $content = json_decode($body, true);
            return response()->json(['response' => $content]);
        } catch (\Exception $exception) {
            Log::warning('[Eiffel API Service] : Error HTTP Response', [
                'message' => $exception->getMessage()
            ]);
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function setMatricule(string $matricule): self
    {
        $this->eiffelApiMatricule = $matricule;

        return $this;
    }

    private function getResponse(string $action, ?string $date): Response
    {
        $payload = [
            'action' => $action,
            'matricule' => $this->eiffelApiMatricule,
            'user' => $this->eiffelApiUser,
            'pass' => $this->eiffelApiPassword,
        ];
        if ($date !== null) {
            $payload['setDate'] = $date;
        }
        if (!empty($this->extraParameters)) {
            $payload = array_merge($payload, $this->extraParameters);
        }
        return Http::withOptions([
            'verify' => false,
            'timeout' => 5,
            'cert' => [
                $this->eiffelCertPath,
                $this->eiffelCertPwd
            ],
            'ssl_key' => $this->eiffelKeyPath,
        ])->get($this->eiffelApiBaseUri, $payload);
    }
}
