<?php

namespace Inside\SETE\Services;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Inside\SETE\Exceptions\EiffelPageException;

class EiffelPageService
{
    public const PLANNING = 'planning';
    public const REAL_TIME_INFO = 'realtimeinfo';

    private array $presenter = [
      self::REAL_TIME_INFO => [
          'direct_tower' => [
              'temp',
              'vent',
              'pnord',
              'pnordAscenseur',
              'pest',
              'pestAscenseur',
              'pkiosk',
              'pkioskAscenseur',
              'psud',
              'psudAscenseur',
          ],
          'fj_accueil' => [
              "daily_sheet_1",
              "daily_sheet_2",
              "daily_sheet_3",
              "daily_sheet_4",
              "daily_sheet_5"
          ],
          'fj_caisses' => [
              "daily_sheet_1",
              "daily_sheet_2",
              "daily_sheet_3",
              "daily_sheet_4",
              "daily_sheet_5"
          ],
          'fj_aes' => [
              "daily_sheet_1",
              "daily_sheet_2",
              "daily_sheet_3",
              "daily_sheet_4",
              "daily_sheet_5"
          ],
          'waiting_time_pillar' => [
              'pnordAffluence',
              'pestAffluence',
              'pkioskAffluence',
              'psudAffluence',
          ]
      ]
    ];
    private string $date;
    private string $page;

    public function __construct(
        private EiffelApiService $eiffelApiService,
    ) {
    }

    public function execute(): array
    {
        $cacheKey = $this->cacheKey();
        $cachedData = Cache::tags(['sete_activites_page'])->get($cacheKey);
        if (!empty($cachedData)) {
            return $cachedData;
        }
        return $this->fetchPageData();
    }

    public function setPage(string $page): self
    {
        $this->page = $page;
        return $this;
    }

    public function setDate(string $date): self
    {
        $this->date = Carbon::parse($date)->format('Y-m-d');
        return $this;
    }

    private function cacheKey(): string
    {
        return sprintf("%s_%s", $this->page, $this->date);
    }

    private function fetchPageData(): array
    {
        try {
            $fetchedPageData = [];
            collect(config('exploitation.endpoints.'.$this->page))->each(
                function (array $fields, string $endpoint) use (&$fetchedPageData) {
                    $values = in_array($endpoint, config('exploitation.daily_sheet'))
                        ? $this->getDailySheets($endpoint) : $this->getValuesFromApi($endpoint, $fields);

                    $fetchedPageData = array_merge($fetchedPageData, $values);
                }
            );

            if (self::PLANNING == $this->page) {
                $this->addVisitors($fetchedPageData);
            }

            Cache::tags(['sete_activites_page'])->put(
                $this->cacheKey(),
                $fetchedPageData,
                config('exploitation.cache.expiration')
            );
        } catch (EiffelPageException $exception) {
            Log::warning('[Eiffel Page Service] : Error ', [
                'message' => $exception->getMessage()
            ]);
        }
        return $fetchedPageData;
    }

    private function addVisitors(array &$fetchedPageData): void
    {
        $visitorFields = collect(['Y', 'Y-1', 'cumul_Y', 'cumul_Y-1']);
        if ($visitorFields->some(fn ($key) => !isset($fetchedPageData[$key]))) {
            return;
        }

        $fetchedPageData['visiteurs_cumul'] = [
            ['annee' => $fetchedPageData['Y'],'nb_visiteurs' => $fetchedPageData['cumul_Y']],
            ['annee' => $fetchedPageData['Y-1'],'nb_visiteurs' => $fetchedPageData['cumul_Y-1']],
        ];

        $visitorFields->each(function ($key) use (&$fetchedPageData) {
            unset($fetchedPageData[$key]);
        });
    }

    private function getDailySheets(string $endpoint): array
    {
        $baseUrl = config('app.url')."/api/v1/sete/download/";
        return [
            $endpoint => [
                "daily_sheet_1" => "{$baseUrl}{$endpoint}",
                "daily_sheet_2" => "{$baseUrl}{$endpoint}?date=" . now()->addDay()->format('Y-m-d'),
                "daily_sheet_3" => "{$baseUrl}{$endpoint}?date=" . now()->addDays(2)->format('Y-m-d'),
                "daily_sheet_4" => "{$baseUrl}{$endpoint}?date=" . now()->addDays(3)->format('Y-m-d'),
                "daily_sheet_5" => "{$baseUrl}{$endpoint}?date=" . now()->addDays(4)->format('Y-m-d'),
            ]
        ];
    }

    private function getValuesFromApi(string $endpoint, array $fields): array
    {
        $response = $this->eiffelApiService->getAction($endpoint, $this->date);
        $data = json_decode($response->content(), true);

        $return = collect($fields)->mapWithKeys(function ($field) use ($data, $endpoint) {
            return ($field === '*') ? [$endpoint => $data['response'] ?? null] :
            [$field => $data['response'][$field] ?? null];
        })->toArray();

        return empty($this->presenter[$this->page]) ? $return : $this->formatValuesPresentation($return);
    }

    private function formatValuesPresentation(array $data): array
    {
        $this->presenter[$this->page] = collect($this->presenter[$this->page])
            ->map(fn ($items) => collect($items)->mapWithKeys(function ($value, $key) use ($data) {
                if (is_string($value) && array_key_exists($value, $data)) {
                    return [$value => $data[$value]];
                }
                return [is_int($key) ? $value : $key => $value];
            })->toArray())
            ->toArray();

        return $this->presenter[$this->page];
    }
}
