<?php

namespace Inside\RHLH\Http\Controllers;

use Drupal\Core\Entity\EntityStorageException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Validation\Rule;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\RHLH\Models\RegistrationNumber;
use Inside\Support\Str;
use Laravel\Lumen\Routing\Controller;

class RegistrationController extends Controller
{
    public function store(Request $request): JsonResponse
    {
        if (Auth::check()) {
            redirect('/');
        }

        $this->validate($request, [
            'email' => ['required', 'email', 'unique:inside_content_users,email'],
            'firstname' => ['required', 'string', 'min:3', 'max:255'],
            'lastname' => ['required', 'string', 'min:3', 'max:255'],
            'registration_number' => [
                'required',
                'numeric',
                'unique:inside_content_users,registration_number',
                'exists:rhlh_registration_numbers,registration_number',
            ],
        ]);

        $identification = RegistrationNumber::where('registration_number', $request->input('registration_number'))->first();

        if (! $identification) {
            return response()->json(['manager' => [Lang::get('registration.number.doesntexist')]], 422);
        }

        $manager = Users::where('email', $identification->manager_email)->first();

        if (! $manager) {
            return response()->json(['manager' => [Lang::get('registration.manager.doesntexist')]], 422);
        }

        try {
            (new BridgeContent())->contentInsert('users', [
                'name' => $request->input('firstname') . ' ' . $request->input('lastname'),
                'email' => Str::lower($request->input('email')),
                'firstname' => $request->input('firstname'),
                'lastname' => $request->input('lastname'),
                'send_mail' => false,
                'status' => true,
                'hotel_manager' => $manager->uuid,
                'registration_status' => 'pending',
                'is_manager' => $identification->is_manager,
                'registration_number' => $identification->registration_number,
            ]);
        } catch (EntityStorageException $exception) {
            return response()->json(['firstname' => [Lang::get('registration.invalid.name')]], 422);
        }

        return response()->json([], 201);
    }

    public function update(Request $request, string $uuid): JsonResponse
    {
        $this->validate($request, [
            'registration_status' => ['required', Rule::in(['pending', 'declined', 'accepted'])],
        ]);

        (new BridgeContent())->contentUpdate('users', [
            'uuid' => $uuid,
            'status' => in_array($request->input('registration_status'), ['pending', 'accepted']),
            'registration_status' => $request->input('registration_status'),
            'send_mail' => true,
        ]);

        return response()->json(['success' => true]);
    }
}
