<?php

declare(strict_types=1);

namespace Inside\PPLH\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\PAndP;
use Inside\Content\Models\Contents\PAndPRegions;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;
use Inside\PPLH\Contracts\PAndPRegionsGenerator;
use Inside\PPLH\Models\PAndPContact;
use Inside\Slug\Facades\ContentUrlGenerator;
use Inside\Workflow\Events\ProposalReviewedEvent;
use Inside\Workflow\Facades\Proposal;
use Inside\Workflow\Models\Proposal as ProposalModel;

class PAndPReviewedListener extends BaseNotificationListener
{
    public function __construct(
        private PAndPRegionsGenerator $regionsGenerator
    ) {
    }

    public function handle(ProposalReviewedEvent $event): void
    {
        $model = $event->proposable;

        if (! $model instanceof PAndP || ! $model->uuid) {
            return;
        }

        $proposal = Proposal::getFromContent($model->uuid, get_class($model));

        if (! $proposal instanceof ProposalModel || $proposal->status !== 1) {
            return;
        }

        $isUpdate = $model->reversePAndPRegions->isNotEmpty();

        $this->regionsGenerator->generateForPAndP($model);

        $this->sendCorpoPublishNotification($model);

        if ($isUpdate) {
            $this->sendCorpoUpdateNotification($model);
            $this->sendRegionUpdateNotifications($model);
        }
    }

    protected function sendCorpoPublishNotification(PAndP $model): void
    {
        $translation = $model->getTranslationIfExists($model->langcode === 'fr' ? 'en' : 'fr');

        $oldContacts = PAndPContact::whereIn('content_uuid', [$model->uuid, $translation->uuid])
            ->pluck('user_uuid')
            ->unique();

        PAndPContact::whereIn('content_uuid', [$model->uuid, $translation->uuid])->delete();

        if ($model->contact->isEmpty()) {
            return;
        }

        /** @var NotificationType $corpoContactType */
        $corpoContactType = NotificationType::query()->where('type', 'system')->where('model', PAndP::class)->where('action', 'p_and_p_published')->first();

        /** @var Users $contact */
        foreach ($model->contact as $contact) {
            $ppContact = PAndPContact::firstOrCreate([
                'content_uuid' => $model->uuid,
                'user_uuid' => $contact->uuid,
            ]);

            if ($oldContacts->contains($ppContact->user_uuid)) {
                continue;
            }

            $this->route = $ppContact->user;
            $this->notify(
                $corpoContactType,
                $ppContact->user,
                $model,
                ['url' => ContentUrlGenerator::generateUrl($model), 'title' => $model->title]
            );

            if ($translation->langcode !== $model->langcode) {
                $ppContact = PAndPContact::firstOrCreate([
                    'content_uuid' => $translation->uuid,
                    'user_uuid' => $contact->uuid,
                ]);

                $this->route = $ppContact->user;
                $this->notify(
                    $corpoContactType,
                    $ppContact->user,
                    $translation,
                    ['url' => ContentUrlGenerator::generateUrl($translation), 'title' => $translation->title]
                );
            }
        }
    }

    protected function sendCorpoUpdateNotification(PAndP $model): void
    {
        $corpoContactType = NotificationType::query()
            ->where('type', 'system')
            ->where('model', PAndP::class)
            ->where('action', 'p_and_p_updated_lhg')
            ->first();

        if (! $corpoContactType instanceof NotificationType) {
            return;
        }

        foreach ($model->contact as $contact) {
            $contact = User::find($contact->uuid);

            if (! $contact instanceof User) {
                continue;
            }

            $this->route = $contact;
            $this->notify($corpoContactType, $contact, $model, ['url' => ContentUrlGenerator::generateUrl($model), 'title' => $model->title]);

            $langcode = $model->langcode === 'fr' ? 'en' : 'fr';
            $translation = $model->getTranslationIfExists($langcode);

            if ($translation->langcode !== $model->langcode) {
                $this->notify($corpoContactType, $contact, $translation, ['url' => ContentUrlGenerator::generateUrl($translation), 'title' => $translation->title]);
            }
        }
    }

    protected function sendRegionUpdateNotifications(PAndP $model): void
    {
        $regions = $model->reversePAndPRegions;

        if ($regions->isEmpty()) {
            return;
        }

        $regionContactType = NotificationType::query()
            ->where('type', 'system')
            ->where('model', PAndP::class)
            ->where('action', 'p_and_p_updated')
            ->first();

        $regionManagerContactType = NotificationType::query()
            ->where('type', 'system')
            ->where('model', PAndP::class)
            ->where('action', 'p_and_p_updated_manager')
            ->first();

        $langcode = $model->langcode === 'fr' ? 'en' : 'fr';
        $translation = $model->getTranslationIfExists($langcode);

        /** @var PAndPRegions $region */
        foreach ($regions as $region) {
            $contact = User::find($region->contact->first()?->uuid);

            if ($regionContactType instanceof NotificationType && $contact instanceof User) {
                $this->route = $contact;
                $this->notify($regionContactType, $contact, $model, ['url' => ContentUrlGenerator::generateUrl($model), 'title' => $model->title]);

                if ($translation->langcode !== $model->langcode) {
                    $this->notify($regionContactType, $contact, $translation, ['url' => ContentUrlGenerator::generateUrl($translation), 'title' => $translation->title]);
                }
            }

            $manager = User::find($region->regionalOffices->first()?->manager?->first()?->uuid);

            if ($regionManagerContactType instanceof NotificationType && $manager instanceof User) {
                $contactName = null;

                if ($contact instanceof User) {
                    $contactName = trim($contact->firstname.' '.$contact->lastname);
                }

                $this->route = $manager;
                $this->notify($regionManagerContactType, $manager, $model, ['url' => ContentUrlGenerator::generateUrl($model), 'title' => $model->title, 'contact' => $contactName]);

                if ($translation->langcode !== $model->langcode) {
                    $this->notify($regionManagerContactType, $manager, $translation, ['url' => ContentUrlGenerator::generateUrl($translation), 'title' => $translation->title, 'contact' => $contactName]);
                }
            }
        }
    }
}
