<?php

declare(strict_types=1);

namespace Inside\PPLH\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentUpdatedEvent;
use Inside\Content\Models\Contents\PAndP;
use Inside\Content\Models\Contents\PAndPRegions;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\Notification;
use Inside\Notify\Models\NotificationType;
use Inside\Slug\Facades\ContentUrlGenerator;

class PAndPRegionUpdatedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ContentUpdatedEvent $event): void
    {
        $model = $event->model;

        if (! $model instanceof PAndPRegions) {
            return;
        }

        $contact = $model->contact->first();

        if (! $contact instanceof Users) {
            return;
        }

        $type = NotificationType::query()
            ->where('type', 'system')
            ->where('model', PAndPRegions::class)
            ->where('action', 'p_and_p_region_assigned')
            ->first();

        if (! $type instanceof NotificationType) {
            return;
        }

        $last = Notification::query()
            ->where('notification_type_id', $type->id)
            ->where('notifiable_uuid', $model->uuid)
            ->orderBy('created_at', 'desc')
            ->first();

        if ($last instanceof Notification && $last->user_uuid === $contact->uuid) {
            return;
        }

        $this->notifyRegionContact($type, $model, $contact);
        $this->notifyCorpoContact($model, $contact);
    }

    private function notifyRegionContact(NotificationType $type, PAndPRegions $model, Users $contact): void
    {
        $pp = $model->pAndP->first();
        $contact = User::find($contact->uuid);

        if (! $contact instanceof User || ! $pp instanceof PAndP) {
            return;
        }

        $this->route = $contact;
        $this->notify($type, $contact, $model, ['url' => ContentUrlGenerator::generateUrl($pp), 'title' => $pp->title]);
    }

    private function notifyCorpoContact(PAndPRegions $model, Users $regionContact): void
    {
        $pp = $model->pAndP->first();
        $type = NotificationType::query()
            ->where('type', 'system')
            ->where('model', PAndPRegions::class)
            ->where('action', 'p_and_p_region_changed')
            ->first();

        if (! $type instanceof NotificationType || ! $pp instanceof PAndP) {
            return;
        }

        $data = [
            'url' => ContentUrlGenerator::generateUrl($pp),
            'title' => $pp->title,
            'contact' => trim($regionContact->firstname.' '.$regionContact->lastname),
            'regionalOffice' => $model->regionalOffices->first()?->title,
        ];

        foreach ($pp->contact as $contact) {
            $contact = User::find($contact->uuid);

            if (! $contact instanceof User) {
                continue;
            }

            $this->route = $contact;
            $this->notify(
                $type,
                $contact,
                $model,
                $data
            );
        }
    }
}
