<?php

declare(strict_types=1);

namespace Inside\PPLH\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\PAndP;
use Inside\Content\Models\Contents\PAndPRegions;
use Inside\Notify\Events\NotificationCreatedEvent;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

class PAndPNotificationCreatedListener extends BaseNotificationListener implements ShouldQueue
{
    private const EXPECTED_NOTIFICATION_ACTIONS = [
        'p_and_p_region_created',
        'p_and_p_region_assigned',
        'p_and_p_region_changed',
        'p_and_p_updated',
        'p_and_p_updated_manager',
        'p_and_p_published',
    ];

    public function handle(NotificationCreatedEvent $event): void
    {
        $notification = $event->notification;
        $notificationType = $notification->trigger;

        if ($notificationType->via !== 'web' || ! in_array($notificationType->action, self::EXPECTED_NOTIFICATION_ACTIONS)) {
            return;
        }

        $key = null;
        $edit = false;

        /** @var PAndP|PAndPRegions $model */
        $model = $notification->notifiable;

        $extras = [];

        switch ($notificationType->action) {
            case 'p_and_p_published':
                $key = 'published';
                break;
            case 'p_and_p_region_created':
                $key = 'region';
                /** @var PAndPRegions $notifiable */
                $notifiable = $notification->notifiable;
                $model = $notifiable->pAndP->first();
                break;
            case 'p_and_p_region_assigned':
                $key = 'region_assigned';
                /** @var PAndPRegions $notifiable */
                $notifiable = $notification->notifiable;
                $model = $notifiable->pAndP->first();
                break;
            case 'p_and_p_region_changed':
                $key = 'region_changed';
                /** @var PAndPRegions $notifiable */
                $notifiable = $notification->notifiable;
                $model = $notifiable->pAndP->first();
                $extras['contact'] = $notification->data['contact'];
                $extras['regionalOffice'] = $notification->data['regionalOffice'];
                $edit = true;
                break;
            case 'p_and_p_updated':
                $key = 'updated';
                $edit = true;
                break;
            case 'p_and_p_updated_manager':
                $key = 'updated_manager';
                $edit = true;

                if (array_key_exists('contact', $notification->data) && ! empty($notification->data['contact'])) {
                    $key = 'updated_manager_with_contact';
                    $extras['contact'] = $notification->data['contact'];
                }
                break;
        }

        if (empty($key)) {
            return;
        }

        $emailNotificationType = NotificationType::query()->where('action', 'p_and_p_email')->first();

        if (! $emailNotificationType instanceof NotificationType) {
            return;
        }

        $to = User::find($notification->user_uuid);

        if (! $to instanceof User || $to->langcode !== $model->langcode) {
            return;
        }

        $mailData = [
            'extra' => [
                'mail' => [
                    'subject' => "pplh.mail.$model->content_type.$key.subject",
                    'text' => "pplh.mail.$model->content_type.$key.content",
                    'buttonText' => "pplh.mail.$model->content_type.button",
                ],
            ],
        ];

        if ($edit) {
            $mailData['url'] = "edit/$model->content_type/$model->uuid";
        }

        $mailData = array_merge($mailData, $extras);

        $this->route = $to;
        $this->notify($emailNotificationType, $to, $model, $mailData);
        // send mail to $notification->user->email based on $notification->notifiable->langcode
    }
}
