<?php

declare(strict_types=1);

namespace Inside\PPLH\Jobs;

use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\PAndP;
use Inside\Content\Models\Contents\PAndPFrance;
use Inside\Content\Models\Contents\RegionalOffices;
use Inside\Jobs\Job;
use Inside\PPLH\Events\PAndPCSVGeneratedEvent;
use Inside\Slug\Facades\ContentUrlGenerator;
use League\Csv\Writer;

class GeneratePAndPCSV extends Job
{
    public const CSV_DOWNLOAD_FOLDER = 'csv/download';

    public function __construct(protected User $user)
    {
    }

    public function handle(): void
    {
        $regions = RegionalOffices::query()
            ->where('langcode', $this->user->langcode)
            ->where('status', 1)
            ->get();

        $array = [];
        $header = [
            'url' => Lang::get('pplh.p_and_p.export.url', [], $this->user->langcode),
            'title' => Lang::get('pplh.p_and_p.export.title', [], $this->user->langcode),
            'p_and_p_categories' => Lang::get('pplh.p_and_p.export.category', [], $this->user->langcode),
            'brands' => Lang::get('pplh.p_and_p.export.brands', [], $this->user->langcode),
            'created_at' => Lang::get('pplh.p_and_p.export.date', [], $this->user->langcode),
            'contact' => Lang::get('pplh.p_and_p.export.lhg_contact', [], $this->user->langcode),
        ];

        foreach ($regions as $region) {
            $header['regions.'.$region->uuid] = Lang::get('pplh.p_and_p.export.region_contact', ['region' => $region->title], $this->user->langcode);
        }

        $array[] = $header;

        $pps = PAndP::query()
            ->where('status', 1)
            ->where('langcode', $this->user->langcode)
            ->get();

        $ppsFrance = PAndPFrance::query()
            ->where('status', 1)
            ->where('langcode', $this->user->langcode)
            ->get();

        $contents = $pps->merge($ppsFrance);

        /** @var PAndP|PAndPFrance $content */
        foreach ($contents as $content) {
            $brands = $content->brands->pluck('title')->toArray();

            $line = [
                'url' => url(ContentUrlGenerator::generateUrl($content)),
                'title' => trim($content->title, " \t\n\r\0\x0B\xC2\xA0"),
                'p_and_p_categories' => trim($content->pAndPCategories->first()->title, " \t\n\r\0\x0B\xC2\xA0"),
                'brands' => trim(implode(' - ', $brands), " \t\n\r\0\x0B\xC2\xA0"),
                'created_at' => date('Y-m-d H:i:s', $content->created_at),
                'contact' => trim($content->contact->first()->email, " \t\n\r\0\x0B\xC2\xA0"),
            ];

            foreach ($regions as $region) {
                $line['regions.'.$region->uuid] = '';
            }

            if ($content instanceof PAndP) {
                $contentRegions = $content->reversePAndPRegions;

                foreach ($contentRegions as $region) {
                    if (! array_key_exists('regions.'.$region->regionalOffices->first()->uuid, $header)) {
                        continue;
                    }

                    $line['regions.'.$region->regionalOffices->first()->uuid] = $region->contact->count() > 0 ? trim($region->contact->first()->email, " \t\n\r\0\x0B\xC2\xA0") : '';
                }
            }

            $array[] = $line;
        }

        Storage::disk('protected')->makeDirectory(self::CSV_DOWNLOAD_FOLDER);

        $filePath = self::CSV_DOWNLOAD_FOLDER.'/p_and_p_'.date('Y_m_d_H_i_s').'.csv';

        Writer::createFromPath(Storage::disk('protected')->path($filePath), "w+")
            ->setDelimiter(';')
            ->insertAll($array);

        PAndPCSVGeneratedEvent::dispatch($this->user->uuid, $filePath);
    }
}
