<?php

namespace Inside\PPLH\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\News;
use Inside\Content\Models\Contents\Spaces;
use Inside\Content\Models\Contents\Tools;
use Inside\Permission\Facades\PermissionSchema;
use Symfony\Component\HttpFoundation\Response;

class PandPRegionsPermissionsMiddleware
{
    private const PATH = 'api/v1/content/p_and_p_regions';

    /**
     * @param Request $request
     * @param Closure $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next): mixed
    {
        /**
         * @var User $user
         */
        $user = Auth::user();

        $skip = !str($request->path())->contains(self::PATH);
        $skip |= $request->getMethod() !== 'GET';
        $skip |= (!$user instanceof User || $user->hasRole('super_administrator') || !$user->hasRole('Régions'));

        if ($skip) {
            return $next($request);
        }


        $roles = $user->roles->pluck('id', 'name')->toArray();

        $response = $next($request);
        if ($response instanceof Response && !$response->isOk()) {
            return $response;
        }

        $regionalOfficesUuids = $user->information->regionalOffices->pluck('uuid');
        $tree = collect(PermissionSchema::tree($roles));
        $content = $this->permissionsFixedContent($response, $regionalOfficesUuids, $tree);

        set_response($response, $content);
        return $response;
    }


    private function permissionsFixedContent(Response $response, Collection $regionalOfficesUuids, Collection $tree): array
    {
        $pAndPRegions = json_decode_response($response);

        $categories = collect($tree->where('type', 'categories')->first());
        $permissions = collect($categories->get('permissions'))
            ->where('type', 'p_and_p')
            ->first();

        $canUpdate = collect($permissions['permission'])
            ->where('action', 'update');

        $pAndPRegionsData = collect($pAndPRegions['data']);

        $pAndPRegionsData->transform(function ($region) use ($canUpdate, $regionalOfficesUuids) {
            if (empty($region['regional_offices'])) {
                return $region;
            }

            $region['admin']['permission']['update'] =
                $canUpdate->isNotEmpty() &&
                collect($region['regional_offices']['data'])
                    ->pluck('uuid')
                    ->every(
                        fn ($uuid) => $regionalOfficesUuids->contains($uuid)
                    );

            return $region;
        });

        $pAndPRegions['data'] = $pAndPRegionsData->toArray();
        return $pAndPRegions;
    }
}
