<?php

namespace Inside\ORSY\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContent;
use Symfony\Polyfill\Intl\Normalizer\Normalizer;

class DirectoryImageSeeder extends Seeder
{
    private BridgeContent $bridgeContent;

    public function run(): void
    {
        $this->bridgeContent = new BridgeContent();
        $photoDirectoryPath = "photos";
        if (!Storage::disk('local')->exists($photoDirectoryPath)) {
            throw new \Exception("Photos Directory not found: ");
        }

        $files = Storage::disk('local')->files($photoDirectoryPath);

        if (empty($files)) {
            throw new \Exception("Aucune photo trouvée dans le dossier : {$photoDirectoryPath}");
        }
        foreach ($files as $filePath) {
            $data = $this->extractNameFromFilename(basename($filePath));
            if ($data) {
                $this->updateUser($data, $filePath);
            } else {
                echo "{$filePath} → Impossible d'extraire le nom et prénom\n";
            }
        }
    }

    private function updateUser(array $data, string $imagePath): void
    {
        $user = type_to_class('users')::where('firstname', $this->normalizeAndLower($data['prenom']))
            ->where('lastname', $this->normalizeAndLower($data['nom']))
            ->get();

        if ($user->count() !== 1) {
            echo sprintf("%s occurences found for %s %s\n", $user->count(), $data['prenom'], $data['nom']);
            ;
            return;
        }

        if (!Storage::disk('local')->exists($imagePath)) {
            echo "Image file not found: $imagePath\n";
            return;
        }

        $payload = [
          'uuid' => $user->first()->uuid,
          'uuid_host' => $user->first()->uuid_host,
          'image' => $imagePath,
        ];

        $userUuid = $this->bridgeContent->contentUpdate(
            type: 'users',
            data: $payload,
            fromCli: true,
            ignoreParagraphs: true
        );
    }

    private function extractNameFromFilename(string $filename): ?array
    {
        $name = (string) preg_replace('/\.[^.]+$/u', '', $filename);
        $name = str_replace('_', ' ', $name);
        $name = preg_replace('/\s+/u', ' ', trim($name));
        if ($name === '' || !is_string($name)) {
            return null;
        }

        if (mb_strpos($name, ' ') !== false) {
            $parts = (array)preg_split('/\s+/u', $name);
        } elseif (mb_strpos($name, '-') !== false) {
            $parts = (array)preg_split('/-+/u', $name);
        } else {
            return null;
        }

        $nomParts = [];
        $prenomParts = [];

        foreach ($parts as $part) {
            if (!is_string($part)) {
                continue;
            }
            $part = trim($part);
            if ($part === '') {
                continue;
            }

            if ($this->isAllUpper($part)) {
                $nomParts[] = $part;
            } else {
                $prenomParts[] = $part;
            }
        }

        if (empty($nomParts) || empty($prenomParts)) {
            return null;
        }

        return [
            'nom'    => implode(' ', $nomParts),
            'prenom' => implode(' ', $prenomParts),
        ];
    }

    private function isAllUpper(string $word): bool
    {
        return $word === mb_strtoupper($word, 'UTF-8');
    }

    private function normalizeAndLower(string $str): string
    {
        $str = Normalizer::normalize($str, Normalizer::FORM_C);
        return mb_strtolower(trim($str), 'UTF-8');
    }
}
