<?php

namespace Inside\OPTI\Services;

use Carbon\Carbon;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Inside\OPTI\Events\GlobalManagersDocumentsUpdatedEvent;
use Inside\OPTI\Events\SpaceManagersDocumentsUpdatedEvent;
use Inside\OPTI\Models\ManagerSpace;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

class ManagerSpacesService
{
    /**
     * @var string
     */
    protected $locale;

    public function __construct()
    {
        $this->locale = App::getLocale();
    }

    private function cleanUrl(string $field, string $column): void
    {
        $row = ManagerSpace::query()->where($column, $field)->first();
        if ($row instanceof ManagerSpace) {
            if (Storage::disk('protected')->exists('managers/' . $row->file)) {
                Storage::disk('protected')->delete('managers/' . $row->file);
            }
            $row->file = null;
            $row->save();
        }
    }

    private function updateTable(string $uuid, string $field, ?string $file, ?string $label): void
    {
        $space = ManagerSpace::query()->where('space_uuid', $uuid)->where('field', $field)->first();
        if ($space instanceof ManagerSpace && ($space->file !== $file || $space->{'label_' . $this->locale} !== $label)) {
            $field === 'stats' ? event(new SpaceManagersDocumentsUpdatedEvent($uuid)) : event(new GlobalManagersDocumentsUpdatedEvent($field));
            $space->file = $file;
            $space->{'label_' . $this->locale} = $label;
            $space->save();
            return;
        } elseif (!$space) {
            ManagerSpace::firstOrCreate(
                [
                    'space_uuid' => $uuid,
                    'field' => $field
                ],
                [
                    'file' => $file,
                    'label_' . $this->locale => $label
                ]
            );
        }
    }

    private function handleDocuments(array $document, string $field, string $column, ?string $path): void
    {
        if ($document['url'] === null) {
            $this->cleanUrl($field, $column);
        } elseif (strpos($document['url'], 'managers/' . $path) === false) {
            if (Storage::disk('protected')->exists('managers/' . $path)) {
                Storage::disk('protected')->delete('managers/' . $path);
            }
            if (!Storage::disk('local')->copy($document['url'], 'protected/managers/' . $path)) {
                return;
            }
        }
        if ($field !== "market" && $field !== "ca") {
            $uuid = $field;
            $field = "stats";
        } else {
            $uuid = "global";
        }
        if (!array_key_exists('label', $document)) {
            $document['label'] = null;
        }
        $this->updateTable($uuid, $field, $path, $document['label']);
    }

    public function addManagerDocs(array $datas): array
    {
        //nothing uploaded
        if (empty($datas)) {
            return [
                "error" => "data is empty"
            ];
        }
        if (array_key_exists('market', $datas) && $datas['market']) {
            $path = $datas['market']['url'] ? 'market.' . pathinfo($datas['market']['url'], PATHINFO_EXTENSION) : null;
            $this->handleDocuments($datas['market'], 'market', 'field', $path);
        }
        if (array_key_exists('ca', $datas) && $datas['ca']) {
            $path = $datas['ca']['url'] ? 'ca.' . pathinfo($datas['ca']['url'], PATHINFO_EXTENSION) : null;
            $this->handleDocuments($datas['ca'], 'ca', 'field', $path);
        }

        foreach ($datas['spaces'] as $chunkedSpace) {
            if (strpos($chunkedSpace['url'], env('APP_STORAGE_PATH')) !== false) {
                continue;
            }

            $path = $chunkedSpace['url'] ? 'space-' . $chunkedSpace['uuid'] . '-stats.' . pathinfo($chunkedSpace['url'], PATHINFO_EXTENSION) : null;
            $this->handleDocuments($chunkedSpace, $chunkedSpace['uuid'], 'space_uuid', $path);
        }
        return [
            "success" => true
        ];
    }

    public function getSpace(array $datas, string $space_uuid): array
    {
        $ca = ManagerSpace::query()->where('space_uuid', 'global')->where('field', 'ca')->select('file', 'label_' . $this->locale)->first();
        $market = ManagerSpace::query()->where('space_uuid', 'global')->where('field', 'market')->select('file', 'label_' . $this->locale)->first();
        $space = ManagerSpace::query()->where('space_uuid', $space_uuid)->where('field', 'stats')->first();

        if ($ca instanceof ManagerSpace) {
            $ca->file = $ca->file ? Storage::disk('protected')->url('managers/' . $ca->file) : null;
            $ca->label = $ca->{'label_' . $this->locale};
            unset($ca->{'label_' . $this->locale});
        }
        if ($market instanceof ManagerSpace) {
            $market->file = $market->file ? Storage::disk('protected')->url('managers/' . $market->file) : null;
            $market->label = $market->{'label_' . $this->locale};
            unset($market->{'label_' . $this->locale});
        }
        $url = $space instanceof ManagerSpace ? Storage::disk('protected')->url('managers/' . $space->file) : null;

        return [
            "ca" => $ca,
            "market" => $market,
            "space" => $url
        ];
    }

    public function getAllSpaces(array $datas): array
    {
        $ca = ManagerSpace::query()->where('space_uuid', 'global')->where('field', 'ca')->select('file', 'label_' . $this->locale)->first();
        $market = ManagerSpace::query()->where('space_uuid', 'global')->where('field', 'market')->select('file', 'label_' . $this->locale)->first();
        $spaces_models = ManagerSpace::query()->where('field', 'stats')->select('space_uuid', 'file')->get();

        if ($ca instanceof ManagerSpace) {
            $ca->file = $ca->file ? Storage::disk('protected')->url('managers/' . $ca->file) : null;
            // front expect variable named label, and not label_fr
            $ca->label = $ca->{'label_' . $this->locale};
            unset($ca->{'label_' . $this->locale});
        }
        if ($market instanceof ManagerSpace) {
            $market->file = $market->file ? Storage::disk('protected')->url('managers/' . $market->file) : null;
            $market->label = $market->{'label_' . $this->locale};
            unset($market->{'label_' . $this->locale});
        }
        $spaces = [];

        foreach ($spaces_models as $model) {
            $elem = $model->getAttributes();

            $elem['file'] = $elem && $elem['file'] ? Storage::disk('protected')->url('managers/' . $elem['file']) : null;
            $spaces[] = $elem;
        }

        return [
            "ca" => $ca,
            "market" => $market,
            "spaces" => $spaces
        ];
    }
}
