import { http } from '@inside-library/helpers/Http'
import { partition } from '@inside-library/helpers/Arrays'

import { CAPSULE_FIELDS } from '../const/news-fields'

export const fetchServiceOffer = async ({ $route }) => {
  try {
    const { data } = await http.get('content/service_offers', {
      fields: [
        'uuid',
        'title',
        'content_type',
        'admin',
        'content',
        'file',
        'updated_at',
        { users: ['uuid', 'image', 'firstname', 'lastname', 'email', 'phone'] },
        { targets: ['uuid', 'title', 'image'] },
        {
          staff_sizes: {
            fields: ['uuid', 'title'],
            filters: { sort: 'title:asc' }
          }
        },
        { branches: ['uuid', 'title'] },
        { offer_themes: ['uuid', 'title', 'color'] },
        {
          regions: {
            fields: ['uuid', 'title', 'color'],
            filters: { 'status:eq': 1 }
          }
        },
        {
          characteristics: {
            fields: ['uuid', 'title'],
            filters: { 'status:eq': 1 }
          }
        },
        {
          youngsters: {
            fields: ['uuid', 'title'],
            filters: { 'status:eq': 1 }
          }
        }
      ],
      filters: { 'slug:eq': $route.params.slug }
    })

    return data[0]
  } catch (err) {
    throw new Error(err)
  }
}

export const fetchRegions = async ({ $route }) => {
  try {
    const { data } = await http.get('content/regions', {
      fields: ['uuid', 'title', 'svg', 'color', 'region_id'],
      filters: { 'status:eq': 1 }
    })

    const { data: custom } = await http.get('content/custom', {
      fields: ['uuid', 'map_introduction'],
      filters: { 'slug:eq': 'agenda' }
    })

    const [dom, metropole] = partition(data, item => item.region_id === 'dom')
    return {
      dom,
      metropole,
      mapIntroduction: custom[0].map_introduction
    }
  } catch (err) {
    throw new Error(err)
  }
}

export const fetchFeaturedNews = async () => {
  try {
    const { data: newsCategories } = await http.get('content/news_categories', {
      fields: ['uuid'],
      filters: { 'status:eq': 1 }
    })

    const promises = newsCategories.map(({ uuid }) =>
      http.get('content/news', {
        fields: CAPSULE_FIELDS,
        filters: {
          news_categories: { 'uuid:eq': uuid },
          'status:eq': 1,
          limit: 1,
          sort: 'updated_at:desc'
        }
      })
    )

    const data = await Promise.all(promises)
    return data
      .flatMap(({ data }) => data)
      .sort((a, b) => b.updated_at - a.published_at)
  } catch (err) {
    throw new Error(err)
  }
}

export const fetchEmailings = async ({ $route }) => {
  try {
    const { data: emailings } = await http.get('content/emailings', {
      fields: [
        'uuid',
        'title',
        'content_type',
        'admin',
        'slug',
        'news',
        'date',
        'content',
        'link',
        'published_at',
        'image',
        { regions: ['uuid', 'title'] },
        { emailings_targets: ['uuid', 'title'] },
        { emailings_categories: ['uuid', 'title'] },
        { branches: ['uuid', 'title'] }
      ],
      filters: { 'slug:eq': $route.params.slug }
    })

    return emailings[0] ?? null
  } catch (err) {
    throw new Error(err)
  }
}
