<?php

namespace Inside\MSLH\Services;

use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Permission\Facades\Permission;

/**
 * @class PermissionCustomFaqsService
 *
 * @package Inside\MSLH\Services
 */
class PermissionCustomFaqsService
{
    /**
     * Attribute specific permissions to faqs $uuids to let $roles read it !
     *
     * @param string $uuid
     * @param array $roleToBeGrantedIds
     * @param array $roleToBeRejectedIds
     */
    public function setPermissions(string $uuid, array $roleToBeGrantedIds, array $roleToBeRejectedIds = []): void
    {
        $permissionSchema = [
            'authorizable_type' => type_to_class('faqs'),
            'authorizable_uuid' => $uuid,
            'action' => 'read',
            'invert' => false,
            'children' => true,
        ];
        // grant
        foreach ($roleToBeGrantedIds as $roleId) {
            $permissionSchemaID = $this->getOrCreatePermissionSchema($permissionSchema);

            $this->grantPermissionSchemaToRoleId($permissionSchemaID, $roleId);
            Permission::buildPermissionForRole($roleId);
        }
        // reject
        $permissionSchema['invert'] = true;
        foreach ($roleToBeRejectedIds as $roleId) {
            $permissionSchemaID = $this->getOrCreatePermissionSchema($permissionSchema);

            $this->grantPermissionSchemaToRoleId($permissionSchemaID, $roleId);
            Permission::buildPermissionForRole($roleId);
        }
    }

    /**
     * get or create a permission schema using $schema config
     *
     * @param array $schema
     * @return int
     */
    protected function getOrCreatePermissionSchema(array $schema): int
    {
        try {
            $schemaId = DB::table('inside_permissions_schema')->insertGetId($schema);
        } catch (QueryException $e) {
            $schemaId = DB::table('inside_permissions_schema')->where($schema)->first()->id;
        }

        return $schemaId;
    }

    /**
     * Grant permission schema $schemaId to $roleId
     *
     * @param int $schemaId
     * @param int $roleId
     */
    protected function grantPermissionSchemaToRoleId(int $schemaId, int $roleId): void
    {
        try {
            DB::table('inside_roles_permissions_schema')->insert(
                [
                    'role_id' => $roleId,
                    'permission_schema_id' => $schemaId,
                    'is_content_specific' => true,
                ]
            );
        } catch (QueryException $e) {
            // Do nothing
            Log::error('[CreatedPermissionCustomFaqsListeners] Failed to grant permission schema <' . $schemaId . '> to role <' . $roleId . '>');
        }
    }
}
