import moment from 'moment'
import { markRaw } from 'vue'

import { http } from '@inside-library/helpers/Http'
import { insertIf } from '@inside-library/helpers/Arrays'
import { formatHref } from '@inside-library/helpers/Langs'

import { Button } from 'inside-collection/components'

export default {
  $_i18nRootPath: 'events.FULL',
  components: { Button },
  data() {
    return {
      m_registered: false
    }
  },
  methods: {
    async $_scaffoldOverrides(data, { replaceItem, addItemAfter }) {
      replaceItem('header', {
        data: {
          border: true,
          borderColor: this.getCheckedData(data, ['events_categories', 'color'])
        }
      })

      const registerType = this.getCheckedData(data, [
        'register_forms',
        'custom_form_type'
      ])
      const feedbackType = this.getCheckedData(data, [
        'feedback_forms',
        'custom_form_type'
      ])

      if (!registerType) {
        return
      }

      const participants = await this.$_fetchParticipants(data.uuid, {
        onlyUsers: true,
        contentType: data.content_type
      })

      addItemAfter('place', {
        id: 'participants',
        tag: 'users-summary-ui',
        class: 'item participants header-item',
        label: this.$_tr('custom.participants'),
        hideIf: () => !participants.length,
        data: { heritedData: participants }
      })

      addItemAfter('participants', {
        id: 'subscribe',
        class: 'item subscription',
        children: [
          ...insertIf(!this.$_eventFinished(data), {
            tag: 'router-link',
            class: 'register',
            data: {
              to: formatHref({
                name: this.$_page('EventRegisterPage', data),
                params: this.$_params(data)
              })
            },
            children: [
              { tag: markRaw(Button), value: this.$_tr('custom.register') }
            ]
          }),
          ...insertIf(this.$_eventFinished(data) && feedbackType, {
            tag: 'router-link',
            class: 'register',
            data: {
              to: formatHref({
                name: this.$_page('EventFeedbackPage', data),
                params: this.$_params(data)
              })
            },
            children: [
              { tag: markRaw(Button), value: this.$_tr('custom.feedback') }
            ]
          }),
          {
            tag: 'router-link',
            value: this.$_tr('custom.seeParticipants'),
            class: 'see see-register',
            data: {
              to: formatHref({
                name: this.$_page('EventParticipantsPage', data),
                params: this.$_params(data)
              })
            }
          },
          ...insertIf(this.$_eventFinished(data) && feedbackType, {
            tag: 'router-link',
            class: 'see see-feedbacks',
            value: this.$_tr('custom.seeFeedbacks'),
            data: {
              to: formatHref({
                name: this.$_page('EventFeedbacksListingPage', data),
                params: this.$_params(data)
              })
            }
          })
        ]
      })
    },
    async $_fetchParticipants(
      eventUuid,
      { onlyUsers, contentType } = { onlyUsers: false }
    ) {
      try {
        const { data } = await http.get('content/events_subscriptions', {
          fields: [
            'uuid',
            { authors: ['uuid', 'firstname', 'lastname', 'image'] }
          ],
          filters: { [contentType]: { 'uuid:eq': eventUuid } }
        })

        return onlyUsers ? data.map(({ authors }) => authors) : data
      } catch (err) {
        throw new Error(err)
      }
    },
    $_checkIfeventIsFinished(start, end) {
      const today = moment()
      const endDate = moment(end || start)

      return today.isAfter(endDate)
    },
    $_isEvent(data) {
      return data.content_type === 'events'
    },
    $_page(page, data) {
      return this.$_isEvent(data) ? page : `Team${page}`
    },
    $_params(data) {
      return {
        event: data.slug[0],
        ...(!this.$_isEvent(data) && { slug: data.groups.data[0].slug[0] }),
        ...(this.$_isEvent(data) && { slug: data.slug[0] })
      }
    }
  },
  computed: {
    $_eventFinished() {
      return data =>
        this.$_checkIfeventIsFinished(data.date * 1000, data.end_date * 1000)
    }
  }
}
