<?php

namespace Inside\LIED\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Listeners\NotifyUserOfCompletedExport;
use Inside\Content\Models\Contents\RoomsReservations;
use Inside\Content\Transformers\ContentTransformer;
use Inside\LIED\Exports\RoomsReservationsExport;
use Inside\LIED\Repositories\ReservationsRepository;
use Inside\Permission\Facades\Permission;
use Recurr\Rule;
use Recurr\Transformer\ArrayTransformer;
use Symfony\Component\Console\Exception\LogicException;

class ReservationsService
{
    /**
     * @var ReservationsRepository
     */
    protected $repository;

    /**
     * The content transformer.
     *
     * @var ContentTransformer
     */
    protected $transformer = null;

    /**
     * @var array
     */
    protected $fields = [
        'uuid',
        'content_type',
        'title',
        'start_date',
        'end_date',
        'body',
        'cooldown',
        'frequency',
        'admin',
        'readiness_status',
        'authors' => [
            'uuid',
            'firstname',
            'lastname',
            'image',
        ],
        'internal_users' => [
            'uuid',
            'firstname',
            'lastname',
            'image',
        ],
        'organizer' => [
            'uuid',
            'firstname',
            'lastname',
            'image',
        ],
        'rooms' => [
            'uuid',
            'title',
            'rooms_category' => [
                'uuid',
                'title',
                'color',
            ],
        ],
    ];

    /**
     * ReservationsService constructor.
     *
     * @param ContentTransformer $transformer
     * @param ReservationsRepository $repository
     */
    public function __construct(ContentTransformer $transformer, ReservationsRepository $repository)
    {
        $this->transformer = $transformer;
        $this->repository = $repository;
    }

    /**
     * get reservation list in a simple pagination result
     *
     * @param array $filters
     * @param array $fields
     * @return mixed
     * @throws \Exception
     */
    public function getReservationsList(array $filters = [], array $fields = [])
    {
        Permission::disableAllowedScope();
        $query = $this->repository->getReservationsListQuery($filters);
        ContentHelper::applySortToQuery(
            $query,
            $filters,
            'start_date',
            'asc',
            ['title', 'start_date', 'end_date', 'email']
        );
        /** @var Collection $reservations */
        $reservations = $query->get();
        Permission::enableAllowedScope();

        /** @var Carbon $date */
        $date = $this->repository->getCarbonFromFront($filters['date']);
        $startDate = $date->copy()->setTime(0, 0);
        $endDate = $date->copy()->setTime(23, 59);

        // Filter recusive
        foreach ($reservations as $key => &$reservation) {
            if (is_null($reservation->frequency)) {
                continue;
            }
            try {
                $reservationStartDate = get_date($reservation->start_date);
                $reservationEndDate = get_date($reservation->end_date);
                if (! $reservationStartDate || ! $reservationEndDate) {
                    throw new \Exception('There is a problem with the reservation start date and end date !');
                }

                if ($reservation->cooldown > 0) {
                    $reservationEndDate->addMinutes($reservation->cooldown);
                }
                $rule = new Rule($reservation->frequency, $reservationStartDate, $reservationEndDate);

                $transformer = new ArrayTransformer();
                $starts = $transformer->transform($rule)->startsBetween($startDate, $endDate, true)->toArray();
                $ends = $transformer->transform($rule)->endsBetween($startDate, $endDate, true)->toArray();
                if (empty($starts) && empty($ends)) {
                    $reservations->forget($key);
                } else {
                    $reservationStartDate->setDate($date->year, $date->month, $date->day);
                    $reservation->start_date = $reservationStartDate->timestamp;
                    $reservationEndDate->setDate($date->year, $date->month, $date->day);
                    if ($reservation->cooldown > 0) {
                        // Remove cooldown from end date
                        $reservationEndDate->subMinutes($reservation->cooldown);
                    }
                    $reservation->end_date = $reservationEndDate->timestamp;
                }
            } catch (\Throwable $e) {
                $reservations->forget($key);
            }
        }

        if (strpos($filters['sort'], ':') !== false) {
            [$sort, $direction] = explode(':', $filters['sort'], 2);
            $descending = $direction != 'asc';
        } else {
            $sort = $filters['sort'];
            $descending = false;
        }
        /** @var Request $request */
        $request = request();
        $reservations =
            $reservations->sortBy($sort, SORT_REGULAR, $descending)->forPage($request->get('page'), $filters['limit'])->values();

        $result = collect();
        foreach ($reservations as $reservation) {
            $result[] = $this->transformer->transform($reservation, $this->fields);
        }

        return ContentHelper::getPaginationFromCollection($result, $filters['limit']);
    }

    /**
     * @param \Inside\Content\Models\Contents\RoomsReservations $reservation
     * @return \Illuminate\Support\Collection
     */
    public function getExceptionDatesFromReservation(RoomsReservations $reservation): Collection
    {
        return collect(json_decode($reservation->exceptions, true));
    }

    public function getAllReservationsList(): Collection
    {
        Permission::disableAllowedScope();
        $reservations = $this->repository->getAllReservationsListQuery()->get();
        Permission::enableAllowedScope();

        // Filter recusive
        /** @var RoomsReservations $reservation */
        foreach ($reservations as $key => &$reservation) {
            if (is_null($reservation->frequency)) {
                continue;
            }
            try {
                $reservationStartDate = get_date($reservation->start_date);
                $reservationEndDate = get_date($reservation->end_date);
                if (! $reservationEndDate) {
                    throw new \Exception('There is a problem with the reservation end date !');
                }

                if ($reservation->cooldown > 0) {
                    $reservationEndDate->addMinutes($reservation->cooldown);
                }
                $rule = new Rule($reservation->frequency, $reservationStartDate, $reservationEndDate);

                $transformer = new ArrayTransformer();
                foreach ($transformer->transform($rule) as $date) {
                    if ($date->getIndex() == 1) {
                        continue; // First iteration already added!
                    }
                    $recusiveReservation = clone $reservation;
                    $recusiveReservation->start_date = $date->getStart();
                    $recusiveReservation->end_date = $date->getEnd();
                    $reservations[] = $recusiveReservation;
                }
            } catch (\Throwable $e) {
                $reservations->forget($key);
            }
        }

        return $reservations->sortBy('start_date')->values();
    }

    public function export(): bool
    {
        $excelPath = 'exports/rooms_reservations/'.now()->format('YmdHis').'_rooms_reservations_'.Str::random(8)
            .'_export.xls';

        $notifications = [];

        /** @var Request $request */
        $request = request();

        if ($request->user()) {
            $notifications[] = new NotifyUserOfCompletedExport($request->user(), $excelPath, 'protected');
        }
        (new RoomsReservationsExport())->queue(
            $excelPath,
            'protected'
        )->chain(
            [
                new NotifyUserOfCompletedExport($request->user(), $excelPath, 'protected'),
            ]
        )->allOnQueue(get_low_priority_queue_name());

        return true;
    }
}
