<?php

namespace Inside\LIED\Repositories;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\App;
use Inside\Content\Models\Contents\RoomsReservations;
use Inside\Database\Eloquent\Builder;
use Inside\LIED\Models\ReservationStatus;
use Throwable;

/**
 * Class ReservationsRepository
 *
 * @package Inside\LIED\Repositories
 */
class ReservationsRepository
{
    /**
     *
     * @param array $filters
     * @return Builder
     */
    public function getReservationsListQuery(array $filters = []): Builder
    {
        // Prepare filters
        $date = $this->getCarbonFromFront($filters['date']);
        if ($date === null) {
            throw new \Exception('date parameter is missing');
        }
        $startDate = $date->copy()->setTime(0, 0);
        $endDate   = $date->copy()->setTime(23, 59);

        $roomUuid = $filters['room_uuid'] ?? null;
        $status   = $filters['readiness_status'] ?? null;
        $langcode = $filters['langcode'] ?? App::getLocale();

        $query = RoomsReservations::addSelect(
            [
                'readiness_status' => ReservationStatus::selectRaw(
                    'CASE WHEN COUNT(*) > 0 THEN \'ready\' ELSE \'todo\' END'
                )->whereColumn('room_reservation_id', 'inside_content_rooms_reservations.uuid')->whereReady(true)->where(
                    function ($query) use ($date) {
                        $query->whereNull('date')->orWhereDate('date', '=', $date->format('Y-m-d'));
                    }
                ),
            ]
        )->with(
            [
                'rooms' => function ($query) {
                    return $query->where('status', true);
                },
            ]
        )->where('langcode', $langcode)->where(
            function ($query) use ($startDate, $endDate) {
                $query->whereNotNull('frequency')->orWhereBetween('start_date', [$startDate, $endDate])
                          ->orWhereBetween(
                              'end_date',
                              [$startDate, $endDate]
                          )->orWhere(
                              function ($query) use ($startDate, $endDate) {
                                  $query->whereDate('start_date', '<=', $startDate)->whereDate(
                                      'end_date',
                                      '>=',
                                      $endDate
                                  );
                              }
                          );
            }
        );
        if (isset($roomUuid)) {
            $query->whereHas(
                'rooms',
                function ($query) use ($roomUuid) {
                    $query->where(type_to_table('rooms') . '.uuid', $roomUuid);
                }
            );
        }
        if ($status !== null) {
            $query->having('readiness_status', '=', $status);
        }
        $query->where(type_to_table('rooms_reservations') . '.status', true);

        return $query;
    }

    /**
     *
     *
     * @return Builder
     */
    public function getAllReservationsListQuery(): Builder
    {
        return RoomsReservations::with(
            [
                'rooms' => function ($query) {
                    return $query->whereStatus(true);
                },
            ]
        )->where(type_to_table('rooms_reservations') . '.status', true);
    }

    /**
     * Get a carbon date from a front date filter request
     *
     * @param string $date
     * @return \Illuminate\Support\Carbon|null
     */
    public function getCarbonFromFront(string $date): ?Carbon
    {
        try {
            $formattedDate = get_date($date, 'Y-m-d');
            if (!$formattedDate) {
                return null;
            }
            return $formattedDate->setTime(12, 0);
        } catch (Throwable $e) {
        }

        return null;
    }
}
