<?php

namespace Inside\LIED\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Models\Contents\RoomsReservations;
use Inside\LIED\Jobs\WarnUsersBeforeEvent;
use Recurr\Rule;
use Recurr\Transformer\ArrayTransformer;

/**
 * Class PrepareWarnBeforeEvent
 *
 * @package Inside\LIED\Listeners
 */
class PrepareWarnBeforeEvent implements ShouldQueue
{
    public function handle(ContentFullyInsertedEvent $event): void
    {
        $reservation = $event->model;
        if (get_class($reservation) === RoomsReservations::class) {

            // Remove actual warn job
            $jobs = DB::table('inside_jobs')->get();
            foreach ($jobs as $job) {
                $payload = json_decode($job->payload);

                $payloadCommand = unserialize($payload->data->command);
                if ($payloadCommand instanceof WarnUsersBeforeEvent && $payloadCommand->reservation == $reservation) {
                    DB::table('inside_jobs')->where('id', $job->id)->delete();
                }
            }
            $startDate = get_date($reservation->start_date);
            $endDate   = get_date($reservation->start_date);
            if (! $startDate || ! $endDate) {
                throw new \Exception('There is a problem with given dates !');
            }

            // Delay job
            if (config('queue.connections.lts', false)) {
                if ($reservation->frequency === null) {
                    if ($startDate->subMinutes(15)->isFuture()) {
                        WarnUsersBeforeEvent::dispatch($reservation)->delay($startDate->subMinutes(15))->onConnection(
                            'lts'
                        );
                    }
                } else {
                    $rule = new Rule($reservation->frequency, $startDate, $endDate);

                    $transformer = new ArrayTransformer();
                    foreach ($transformer->transform($rule) as $date) {
                        $date = Carbon::instance($date->getStart());
                        if ($date->subMinutes(15)->isFuture()) {
                            WarnUsersBeforeEvent::dispatch($reservation)->delay($date->subMinutes(15))->onConnection(
                                'lts'
                            );
                        }
                    }
                }
            } else {
                if ($startDate->subMinutes(15)->isFuture()) {
                    WarnUsersBeforeEvent::dispatch($reservation)->delay($startDate->subMinutes(15));
                }
            }
        }
    }
}
