<?php

namespace Inside\LIED\Exports;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Lang;
use Inside\Content\Services\Exporters\ContentExportable;
use Inside\LIED\Facades\Reservations;
use Inside\LIED\Models\ReservationStatus;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use PhpOffice\PhpSpreadsheet\Shared\Date;

class RoomsReservationsExport implements FromCollection, WithMapping, WithColumnFormatting, WithHeadings, ShouldAutoSize
{
    use Exportable;
    use ContentExportable;

    public function collection()
    {
        return Reservations::getAllReservationsList();
    }

    public function columnFormats(): array
    {
        return [
            'D' => 'dd mm YYYY',
            'E' => '0.00',
            'F' => 'h:mm',
            'G' => 'h:mm',
        ];
    }

    public function headings(): array
    {
        return [
            Lang::get('reservations.export.header.title'),
            Lang::get('reservations.export.header.description'),
            Lang::get('reservations.export.header.author'),
            Lang::get('reservations.export.header.day'),
            Lang::get('reservations.export.header.duration'),
            Lang::get('reservations.export.header.start_time'),
            Lang::get('reservations.export.header.end_time'),
            Lang::get('reservations.export.header.room'),
            Lang::get('reservations.export.header.reservation_for'),
            Lang::get('reservations.export.header.internal_attendees'),
            Lang::get('reservations.export.header.external_attendees'),
            Lang::get('reservations.export.header.it'),
            Lang::get('reservations.export.header.catering'),
            Lang::get('reservations.export.header.internal_comment'),
        ];
    }

    public function map($row): array
    {
        $startDate = get_date($row->start_date);
        $endDate   = get_date($row->end_date);
        if (! $startDate || ! $endDate) {
            throw new \Exception('Start date and end date are required !');
        }

        $externalUsers = collect($row->external_users !== null ? explode(',', $row->external_users) : []);

        $reservationStatus = ReservationStatus::where('room_reservation_id', $row->uuid)->where(
            function ($query) use ($startDate) {
                $query->where(
                    function ($query) use ($startDate) {
                        $query->whereHas(
                            'reservation',
                            function ($query) {
                                $query->whereNotNull('frequency');
                            }
                        )->whereDate('date', '=', $startDate->format('Y-m-d'));
                    }
                )->orWhere(
                    function ($query) {
                        $query->whereHas(
                            'reservation',
                            function ($query) {
                                $query->whereNull('frequency');
                            }
                        )->whereNull('date');
                    }
                );
            }
        )->first();

        return [
            $row->title,
            $row->reservation_description,
            $row->author !== null ? inside_user_fullname_with_email($row->authors) : 'No author!!!',
            Date::dateTimeToExcel($startDate), //->format('Y-m-d'),
            $endDate->diffInMinutes($startDate),
            $startDate->format('H:i'),
            $endDate->format('H:i'),
            $row->rooms->first()->title ?? 'No room !!',
            $row->organizer !== null && $row->organizer->isNotEmpty() ? inside_user_fullname_with_email(
                $row->organizer->first()
            ) : 'No organizer !!',
            $row->internalUsers !== null && $row->internalUsers->isNotEmpty() ? $row->internalUsers->transform(
                function ($user) {
                    return inside_user_fullname_with_email($user);
                }
            )->implode(',') : '',
            $externalUsers->implode(','),
            $row->informationTechnologies->implode('title', ',') ?? '',
            $row->cateringServices->implode('title', "\n") ?? '',
            ($reservationStatus !== null) ? $reservationStatus->comment : null,
        ];
    }
}
