<?php

namespace Inside\LIED\Notifications;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\AnonymousNotifiable;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Messages\MailMessage;
use Inside\Notify\Models\NotificationType;

class EventsSubscriptionNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * @var mixed
     */
    public $model;

    /**
     * @var User
     */
    public $user;

    /**
     * @var bool
     */
    public $withIcs;

    /**
     * @var NotificationType
     */
    public $notificationType;

    /**
     * @var array
     */
    public $data;

    /**
     * @param NotificationType $notificationType
     * @param Authenticatable|null $user
     * @param mixed $model
     * @param array $data
     */
    public function __construct(NotificationType $notificationType, ?Authenticatable $user, $model, array $data = [])
    {
        /** @var User $currentUser */
        $currentUser = Auth::user();

        $this->notificationType = $notificationType;
        $this->model = $model;
        $this->user = $user instanceof User ? $user : $currentUser;
        $this->withIcs = false;
        $this->data = $data;
    }

    public function withIcs(): self
    {
        $this->withIcs = true;

        return $this;
    }

    /**
     * @param mixed $notifiable
     * @return string[]
     */
    public function via($notifiable): array
    {
        return ['mail'];
    }

    private function fold(string $string): array
    {
        $lines = [];

        if (function_exists('mb_strcut')) {
            while (strlen($string) > 0) {
                if (strlen($string) > 75) {
                    $lines[] = mb_strcut($string, 0, 75, 'utf-8');
                    $string = ' '.mb_strcut($string, 75, strlen($string), 'utf-8');
                } else {
                    $lines[] = $string;
                    $string = '';
                    break;
                }
            }
        } else {
            /** @var array $array */
            $array = preg_split('/(?<!^)(?!$)/u', $string);
            $line = '';
            $lineNo = 0;
            foreach ($array as $char) {
                $charLen = strlen($char);
                $lineLen = strlen($line);
                if ($lineLen + $charLen > 75) {
                    $line = ' '.$char;
                    ++$lineNo;
                } else {
                    $line .= $char;
                }
                $lines[$lineNo] = $line;
            }
        }

        return $lines;
    }

    public function attachIcs(MailMessage $mailMessage, array $data): void
    {
        $startDate = get_date($this->model->date);
        $endDate = get_date($this->model->end_date);
        $created = get_date($this->model->created_at);
        $updated = get_date($this->model->updated_at);

        if (!$startDate || !$endDate || !$created || !$updated) {
            Log::error('[Mail notification] There is a problem with the given dates');
            return;
        }

        $ics = [
            'BEGIN:VCALENDAR',
            'PRODID:-//Inside//Inside Events//EN',
            'VERSION:2.0',
            'CALSCALE:GREGORIAN',
            'METHOD:REQUEST',
            'BEGIN:VEVENT',
            'UID:'.$this->model->uuid,
            'SEQUENCE:'.time(),
                'LOCATION:'.$this->model->place,
            #  'ORGANIZER;SENT-BY="MAILTO:' . $this->model->authors->email . '":MAILTO:'.config('mail.from.address', 'organizer@example.com'),
            'ORGANIZER:MAILTO:'.(($this->user->uuid === $this->model->author) ? config(
                'mail.from.address',
                'organizer@example.com'
            ) : $this->model->authors->email),
            'DTSTART:'.$startDate->format('Ymd\THis\Z'),
            'DTEND:'.$endDate->format('Ymd\THis\Z'),
            'DTSTAMP:'.$created->format('Ymd\THis\Z'),
            'CREATED:'.$created->format('Ymd\THis\Z'),
            'LAST-MODIFIED:'.$updated->format('Ymd\THis\Z'),
            'SUMMARY:'.$this->model->title,
            'END:VEVENT',
            'END:VCALENDAR',
        ];

        $icsLines = [];
        foreach ($ics as $line) {
            foreach ($this->fold($line) as $l) {
                $icsLines[] = $l;
            }
        }

        // Attach to the mail
        $mailMessage->attachData(
            implode("\r\n", $icsLines),
            'invite.ics',
            ['mime' => 'text/calendar; charset=UTF-8; method=REQUEST', 'as' => 'invite.ics']
        );
    }

    protected function getTranslation(string $key, array $data, string $locale): string
    {
        $str = Lang::get($key, $data, $locale);
        $str = str_replace(':title', $this->model->title, $str);

        return $str;
    }

    protected function setLocale(string $locale): void
    {
        $locale = Str::lower($locale);
        $fullLocale = Str::lower($locale).'_'.Str::upper($locale).'.utf8';
        \Illuminate\Support\Facades\Lang::setLocale($locale);
        setlocale(LC_TIME, $locale, $fullLocale);
        \Illuminate\Support\Carbon::setLocale($locale);
        Carbon::setUtf8(false);
    }

    /**
     * @param mixed $notifiable
     * @return MailMessage
     */
    public function toMail($notifiable)
    {
        $data = [];
        $data['to'] = [
            'uuid' => '',
            'lastname' => '',
            'firstname' => '',
            'fullname' => '',
            'image' => '',
        ];
        if (!$notifiable instanceof AnonymousNotifiable && $notifiable->exists) {
            $recipient = Users::find($notifiable->uuid);
            $data['to'] = [
                'uuid' => $recipient->uuid,
                'lastname' => $recipient->lastname,
                'firstname' => $recipient->firstname,
                'fullname' => $recipient->firstname.' '.$recipient->lastname,
                'image' => $recipient->image,
            ];
        }
        $author = Users::find($this->user->uuid);
        if ($author) {
            $data['author'] = [
                'uuid' => $author->uuid,
                'lastname' => $author->lastname,
                'firstname' => $author->firstname,
                'fullname' => $author->firstname.' '.$author->lastname,
                'image' => $author->image,
            ];
        }
        $locale = $notifiable->langcode ?? config('app.locale');
        $subject =
            $this->notificationType->data['mail']['subject'] ?? ($this->data['extra']['mail']['subject'] ?? '');
        $text = $this->data['extra']['mail']['text'] ?? ($this->notificationType->data['mail']['text'] ?? '');
        $buttonText =
            $this->notificationType->data['mail']['buttonText'] ?? ($this->data['extra']['mail']['buttonText'] ?? '');
        $url = null;
        $from = false;
        $this->setLocale($locale);

        Log::warning('[subject]'.$subject.' [text]'.$text.' [button]'.$buttonText);
        $subject = $this->getTranslation(
            $subject,
            $this->data,
            $locale
        );
        $text = $this->getTranslation(
            $text,
            $this->data,
            $locale
        );
        $buttonText = $this->getTranslation(
            $buttonText,
            $this->data,
            $locale
        );

        Log::warning('[subject]'.$subject.' [text]'.$text.' [button]'.$buttonText);

        if (isset($this->data['url']) && !empty($this->data['url'])) {
            $url = $this->data['url'];
        } elseif ((isset($this->model->slug) && is_array($this->model->slug) && count($this->model->slug) > 0)) {
            $url = $this->model->slug[0];
        }

        $mailMessage = (new MailMessage())->markdown('emails.notification', ['locale'])->model($this->model)->subject(
            '["'.env('APP_NAME').'"]'.$subject
        );

        if ($this->withIcs) {
            $this->attachIcs($mailMessage, $data);
        }

        $mailMessage->line($text);

        $mailMessage->action($this->getTranslation($buttonText, [], $locale), config('app.url').'/'.$url);

        return $mailMessage;
    }
}
