<?php

namespace Inside\LIED\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Notification;
use Inside\Authentication\Models\User;
use Inside\LIED\Notifications\ReservationCreatedTeamNotification;
use Inside\Permission\Models\Role;
use Inside\Reservation\Events\ParticipantsOnBookingCreationWarned;

class WarnTeamsOnReservationCreation implements ShouldQueue
{
    /**
     *
     *
     * @param ParticipantsOnBookingCreationWarned $event
     */
    public function handle(ParticipantsOnBookingCreationWarned $event): void
    {
        // To be warned
        $warned = collect();
        $reservation = $event->reservation;
        $notified = collect($event->notified);

        // Warn reception team
        /** @var Role $receptionTeamRole */
        $receptionTeamRole = Role::where(['name' => 'reception_team'])->firstOrFail();
        $warned = $warned->merge($receptionTeamRole->users()->where('status', true)->pluck('uuid'));

        // Warn catering services
        if ($reservation->cateringServices && $reservation->cateringServices->isNotEmpty()) {
            /** @var Role $cateringServicesTeamRole */
            $cateringServicesTeamRole = Role::where(['name' => 'catering_services_team'])->firstOrFail();
            $warned =
                $warned->merge($cateringServicesTeamRole->users()->where('status', true)->pluck('uuid'));
        }

        // Warn information technologies
        if ($reservation->informationTechnologies && $reservation->informationTechnologies->isNotEmpty()) {
            /** @var Role $informationTechnologiesTeamRole */
            $informationTechnologiesTeamRole = Role::where(['name' => 'information_technologies_team'])->firstOrFail();
            $warned =
                $warned->merge($informationTechnologiesTeamRole->users()->where('status', true)->pluck('uuid'));
        }

        // Author
        if ($reservation->authors !== null && !$notified->contains($reservation->authors->email) && !$warned->contains($reservation->authors->uuid)) {
            /** @var ?User $user */
            $user = User::find($reservation->author);
            if ($user) {
                $warned->push($user->uuid);
            }
        }

        // Finally, inform organizer
        if ($reservation->organizer !== null && $reservation->organizer->isNotEmpty() && !$notified->contains($reservation->organizer->first()->email) && !$warned->contains($reservation->organizer->first()->uuid)) {
            /** @var ?User $user */
            $user = User::find($reservation->organizer->first()->uuid);
            if ($user) {
                $warned->push($user->uuid);
            }
        }

        $warned->each(
            function (string $userUuid) use ($reservation, $notified) {
                /** @var ?User $user */
                $user = User::find($userUuid);
                if (!$user || $notified->contains($user->email)) {
                    return;
                }
                Notification::send(
                    $user,
                    (new ReservationCreatedTeamNotification(
                        $reservation,
                        $reservation->creator,
                        env('LIED_RESERVATION_SLUG', 'room-reservations-1')
                    ))->withIcs()
                );
            }
        );
    }
}
