<?php

namespace Inside\LIED\Http\Middleware;

use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use Inside\Content\Models\Contents\RoomsReservations;
use Inside\Content\Facades\ContentHelper;
use Inside\Http\Middleware\TransformResponseAndRequest;
use Inside\LIED\Models\ReservationStatus;

/**
 * Class SaveReadinessStatusAndInternalContentOnReservationEditForm
 *
 * @package Inside\LIED\Http\Middleware
 */
class SaveReadinessStatusAndInternalContentOnReservationEditForm extends TransformResponseAndRequest
{
    protected array $paths = [
        'PUT' => [
            '#^api/v1/content/rooms_reservations/([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})$#',
        ],
        'POST' => [
            'api/v1/content/rooms_reservations',
        ],
    ];

    /**
     * @var bool
     */
    protected $readinessStatus = false;

    /**
     * @var string|null
     */
    protected $internalComment = null;

    /**
     * @var Carbon|null
     */
    protected $date = null;

    /**
     * Save startDate
     *
     * @var Carbon|null
     */
    protected $startDate = null;

    /**
     * Frequency
     *
     * @var string|null
     */
    protected $frequency = null;

    /**
     * @inheritDoc
     */
    protected function patchRequest(Request $request): Request
    {
        $this->readinessStatus = ContentHelper::getAndRemoveQuery($request, 'readiness_status') == 'ready';
        $this->internalComment = ContentHelper::getAndRemoveQuery($request, 'internal_comment');
        $date = ContentHelper::getAndRemoveQuery($request, 'date');
        $this->date = $date !== null ? get_date($date, 'Y-m-d') : null;

        if ($this->date === null) {
            $date = $request->get('start_date');

            if ($date && ($formattedDate = get_date($date, 'Y-m-d H:i:s'))) {
                $this->startDate = $formattedDate->setTime(0, 0, 0);
            } else {
                return $request;
            }
        }

        $this->frequency = $request->get('frequency');

        return $request;
    }

    protected function patch(Request $request, mixed $response, array $data = []): mixed
    {
        if (empty($data)) {
            return $this->patchOnStore($request, $response);
        }

        return $this->patchOnUpdate($request, $response, $data[1]);
    }

    /**
     * create reservation status
     */
    protected function patchOnStore(Request $request, mixed $response): mixed
    {
        /** @var array|null $data */
        $data = json_decode_response($response);
        if ($data === null || !isset($data['uuid'])) {
            Log::error('[patchOnStore] response does not have an uuid !!');

            return $response;
        }
        if ($this->date === null) {
            if ($this->startDate === null) {
                Log::error('[patchOnStore] date is missing from request');

                return $response;
            } else {
                $this->date = $this->startDate;
            }
        }

        ReservationStatus::create(
            [
                'room_reservation_id' => $data['uuid'],
                'date' => $this->frequency !== null ? $this->date : null,
                'ready' => $this->readinessStatus,
                'comment' => $this->internalComment,
                'created_at' => now(),
            ]
        );

        return $response;
    }

    /**
     * Update reservation status
     */
    protected function patchOnUpdate(Request $request, mixed $response, string $uuid): mixed
    {
        /** @var array|null $data */
        $data = json_decode_response($response);
        if ($data === null || $data['uuid'] != $uuid) {
            Log::error('[patchOnUpdate] response does not have an uuid !!');

            return $response;
        }
        $reservation = RoomsReservations::find($uuid);
        if ($reservation === null) {
            Log::error('[patchOnUpdate] reservation ['.$uuid.'] does not exist');

            return $response;
        }

        if ($reservation->frequency !== null) {
            if ($this->date === null) {
                // C'est probablement une suppression d'occurence
                return $response;
            }
            $status =
                ReservationStatus::where('room_reservation_id', $uuid)->whereDate('date', $this->date->format('Y-m-d'))
                    ->first();
        } else {
            $status = ReservationStatus::where('room_reservation_id', $uuid)->whereNull('date')->first();
        }

        if ($status === null) {
            ReservationStatus::create(
                [
                    'room_reservation_id' => $uuid,
                    'ready' => $this->readinessStatus,
                    'comment' => $this->internalComment,
                    'date' => $this->date,
                    'created_at' => now(),
                ]
            );

            return $response;
        }
        $status->fill(
            [
                'ready' => $this->readinessStatus,
                'comment' => $this->internalComment,
                'updated_at' => now(),
            ]
        )->save();

        return $response;
    }
}
