<?php

declare(strict_types=1);

namespace Inside\LHG\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Inside\Content\Models\Content;
use Inside\Workflow\Facades\Proposal as ProposalFacade;
use Inside\Workflow\Facades\Workflow as WorkflowFacade;
use Inside\Workflow\Models\Proposal;
use Inside\Workflow\Models\Workflow;

final class EqualizeContentWorkflow extends Command
{
    /**
     * @var string
     */
    protected $name = 'lhg:workflow:equilize';

    /**
     * @var string
     */
    protected $signature = 'lhg:workflow:equilize';

    /**
     * @var string
     */
    protected $description = 'Equilize every contents in workflow with their translation';

    public function handle(): void
    {
        $configuration = config('louvrehotels.autotranslation');

        if (! $configuration['enabled']) {
            $this->error('Auto-translation is disabled.');

            return;
        }

        Workflow::all()->each(function (Workflow $workflow) {
            $this->info("\nSearching for contents without equilized workflow in {$workflow->title}...");

            $mismatch = $workflow
                ->proposals()
                ->where('status', 2)
                ->get()
                ->map(fn (Proposal $proposal) => $proposal->getProposableModelAttribute())
                ->reject(fn (?Content $content) => ProposalFacade::bypassWorkflow($content))
                ->filter()
                ->groupBy('uuid_host')
                ->reject(fn (Collection $proposals) => $proposals->count() >= 2)
                ->flatten();

            $this->warn("Found {$mismatch->count()} contents without equilized workflow");

            $mismatch->each(function (Content $content) {
                $this->warn("\n    Equilizing content status for {$content->uuid} of {$content->content_type}...");

                $langcode = $content->langcode === 'fr' ? 'en' : 'fr';
                $translation = $content->getTranslationIfExist($langcode);

                if (! WorkflowFacade::isContentInWorkflow($translation)) {
                    $this->warn("    Translating found in {$langcode} without worflow, {$translation->uuid} of {$translation->content_type}...");

                    is_null(ProposalFacade::attachToContent($translation))
                        ? $this->error('    Proposal creation failed, the original content should not be in workflow')
                        : $this->info('    Proposal created successfully');
                }
            });
        });

        $this->info("\nContents workflow equilized successfully!");
    }
}
