<?php

declare(strict_types=1);

namespace Inside\LHG\Console;

use Exception;
use Illuminate\Console\Command;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Facades\Http;
use Inside\Host\Bridge\BridgeContent;
use Inside\LHG\Jobs\InsertContentThroughJob;
use Inside\LHG\Jobs\TranslateArchiveJob;
use Inside\LHG\Jobs\UpdateContentThroughJob;
use Inside\LHG\Services\DuplicateLogicService;
use Laravel\Lumen\Routing\UrlGenerator;

final class DuplicateMissingContent extends Command
{
    /**
     * @var string
     */
    protected $name = 'lhg:duplicate:missing';

    /*
   * @var string
   */
    protected $signature = 'lhg:duplicate:missing';

    /**
     * @var string
     */
    protected $description = 'Duplicate missing content from Louvre Hotels';

    public function handle(): void
    {
        $this->info('Duplicating missing content from Louvre Hotels...');

        $configuration = config('louvrehotels.autotranslation');

        if (! $configuration['enabled']) {
            $this->error('Auto-translation is disabled.');

            return;
        }

        $contentTypes = $configuration['contentTypes'];

        foreach ($contentTypes as $contentType) {
            $this->info("Duplicating missing content for {$contentType}...");

            $relations = Schema::getRelations($contentType);
            $files = Schema::getFieldListingOfType($contentType, ['file', 'image']);

            $uuid_hosts = type_to_class($contentType)::query()
                ->select('uuid_host')
                ->groupBy('uuid_host')
                ->having(DB::raw('count(status)'), '<', 2)
                ->pluck('uuid_host');

            $this->comment("  Found {$uuid_hosts->count()} missing content.\n");

            foreach ($uuid_hosts as $uuid_host) {
                $this->duplicateContent($contentType, $uuid_host, $relations, $files);
            }
        }

        $this->info('Content duplicated successfully!');
    }

    protected function duplicateContent(string $contentType, string $uuid_host, array $relations, array $files): void
    {
        $source = type_to_class($contentType)::query()->with($relations)->where('uuid_host', $uuid_host)->first();

        $langcode = collect(list_languages())->reject($source->langcode)->first();

        $this->warn("  Duplicating content {$source->uuid} to {$langcode}...");

        foreach ($relations as $relation) {
            $data = data_get($source, $relation);

            if (blank($data)) {
                continue;
            }

            $uuids = $data instanceof Collection ? $data->pluck('uuid')->toArray() : $data->uuid;
            $bundle = $data instanceof Collection ? class_to_type($data->first()) : class_to_type($data);

            $translatedUuids = DuplicateLogicService::getEveryTranslatedContent($bundle, $uuids, $langcode);

            data_set($source, Str::snake($relation), $translatedUuids);
        }

        $data = collect($source->getAttributes())
            ->put('bundle', $contentType)
            ->put('type', 'node')
            ->put('langcode', $langcode)
            ->except('uuid', 'pid')
            ->put('content', DuplicateLogicService::generateFreshSections($source->section_content))
            ->merge(collect($files)->mapWithKeys(fn (string $field) => [$field => $source->{$field}]));

        InsertContentThroughJob::dispatch($contentType, $data->toArray())->onQueue(get_high_priority_queue_name())->chain([
            new TranslateArchiveJob($contentType, $source, $langcode),
        ]);
    }
}
