<?php

declare(strict_types=1);

namespace Inside\LHG\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Inside\Host\Bridge\BridgeContent;

final class DeleteCorruptedUsers extends Command
{
    /**
     * @var string
     */
    protected $name = 'lhg:users:delete';

    /**
     * @var string
     */
    protected $signature = 'lhg:users:delete';

    /**
     * @var string
     */
    protected $description = 'Delete corrupted users [NEVER USE IN PRODUCTION]';

    public function handle(): void
    {
        $this->info('Deleting corrupted users...');

        $uuids = DB::table('inside_content_users')
            ->join('inside_users', 'inside_content_users.uuid', '=', 'inside_users.uuid')
            ->join('users', 'inside_users.uuid_host', '=', 'users.uuid')
            ->select('inside_content_users.uuid', 'users.uid')
            ->whereNotNull('api_id')
            ->whereNull('provider_type')
            ->pluck('uid', 'uuid');

        $owner = DB::table('users_field_data')
            ->join('users', 'users_field_data.uid', '=', 'users.uid')
            ->join('inside_users', 'users.uuid', '=', 'inside_users.uuid_host')
            ->where('mail', 'technique@maecia.com')
            ->select('users.uid', 'inside_users.uuid')
            ->first();

        /** @var string $owner_uuid */
        $owner_uuid = $owner->uuid; //@phpstan-ignore-line

        /** @var int $owner_uid */
        $owner_uid = $owner->uid; //@phpstan-ignore-line

        $this->warn('Set author ownership of content to technique');

        // Update the author of the content to the technique user
        $tables = collect(DB::connection()->getDoctrineSchemaManager()->listTableNames()); //@phpstan-ignore-line

        $this->warn('Replacing ownership in section tables');
        $sections = $tables->filter(fn ($table) => str_contains($table, 'inside_section_'));
        $this->replaceOwnershipInTables($sections->toArray(), ['author', 'author_id', 'update_author'], $uuids->keys()->toArray(), $owner_uuid);

        $this->warn('Replacing ownership in content tables');
        $contents = $tables->filter(fn ($table) => str_contains($table, 'inside_content_'))->forget('inside_content_users');
        $this->replaceOwnershipInTables($contents->toArray(), ['author', 'author_id', 'update_author'], $uuids->keys()->toArray(), $owner_uuid);

        $this->warn('Replacing ownership in pivots tables');
        $this->replaceOwnershipInTables(['inside_pivots'], ['parent_uuid', 'related_uuid'], $uuids->keys()->toArray(), $owner_uuid);

        $this->warn('Replacing ownership in revision tables');
        $this->replaceOwnershipInTables(['inside_revisions'], ['user_id'], $uuids->keys()->toArray(), $owner_uid);

        $this->replaceOwnershipInTables([
            'comment_field_data',
            'node_field_revision',
            'node_field_data',
        ], ['uid'], $uuids->values()->toArray(), $owner_uid);

        $this->warn('Remove old users from statistics tables');
        $statistiques = collect(config('statistics.types'))->keys()->map(fn ($type) => type_to_stats_table($type));
        $this->removeFromTables([
            'inside_authentication_statistics',
            'inside_workflow_proposal_steps',
            'inside_users_roles',
            'inside_reactions',
            'inside_statistics',
            'inside_newsletters_statistics',
            'inside_notifications',
            'inside_notifications_subscribers',
            'inside_notifications_users_views',
            'inside_newsletters_sent',
            'content_visits',
            'user_visits',
            ...$statistiques->toArray(),
        ], ['user_uuid'], $uuids->keys()->toArray());

        $this->warn('Remove old users from authentication statistics tables');
        $this->removeFromTables([
            'authentication_statistics',
            'user_statistics',
        ], ['uuid'], $uuids->keys()->toArray());

        $this->warn('Deleting corrupted users');

        // Delete the corrupted users
        $bridge = new BridgeContent();
        $progress = $this->output->createProgressBar($uuids->count());
        $progress->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");

        for ($i = 0; $i < $uuids->count(); $i++) {
            $progress->setMessage("Deleting user {$uuids->keys()[$i]}");
            $bridge->contentDelete('users', $uuids->keys()[$i]);
            $progress->advance();
        }

        $progress->finish();

        $this->info('All corrupted users have been deleted.');
    }

    public function replaceOwnershipInTables(array $tables, array $columns, array $from, string|int $to): void
    {
        foreach ($tables as $table) {
            if (! Schema::hasTable($table)) {
                continue;
            }

            foreach ($columns as $column) {
                try {
                    DB::table($table)->whereIn($column, $from)->update([$column => $to]);
                } catch (\Exception $e) {
                    if ($e->getCode() === '23000') {
                        DB::table($table)->whereIn($column, $from)->delete();

                        continue;
                    }

                    $this->warn("Failed to update column {$column} in table {$table}");
                }
            }
        }
    }

    public function removeFromTables(array $tables, array $columns, array $from): void
    {
        foreach ($tables as $table) {
            if (! Schema::hasTable($table)) {
                continue;
            }

            foreach ($columns as $column) {
                DB::table($table)->whereIn($column, $from)->delete();
            }
        }
    }
}
