<?php

namespace Inside\LHG\Actions;

use Closure;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\LHG\Jobs\UpdateContentThroughJob;
use Inside\LHG\Services\DuplicateLogicService;
use Throwable;

final class TranslateRestoredContent
{
    protected string $bundle;

    protected string $class;

    protected array $relations;

    protected string $currentLangcode;

    protected string $uuid;

    public function __construct(
        protected Request $request,
        private Closure $next
    ) {
    }

    public function handle(): JsonResponse
    {
        if (DuplicateLogicService::isAutoTranslationDisabled()) {
            return $this->next($this->request);
        }

        $this->prepare();

        if (DuplicateLogicService::isNotDuplicable($this->bundle)) {
            return $this->next($this->request);
        }

        return $this->duplicate();
    }

    protected function getCurrentLangcode(): string
    {
        return $this->currentLangcode;
    }

    protected function getTargetLangcode(): string
    {
        return collect(list_languages())->reject($this->getCurrentLangcode())->first();
    }

    /**
     * @throws Throwable
     */
    protected function prepare(): void
    {
        $uri = str($this->request->getPathInfo())->after('revision/');
        $this->bundle = $uri->before('/')->toString();
        $this->uuid = $uri->after($this->bundle.'/')->before('/')->toString();

        $this->class = type_to_class($this->bundle);
        $this->relations = collect(Schema::getRelations($this->bundle))->diff(['authors'])->values()->toArray();
    }

    protected function next(Request $request): mixed
    {
        return ($this->next)($request);
    }

    protected function duplicate(): JsonResponse
    {
        Log::info('[LHG Commons] TranslateRestoredContent: Starting translation of '.$this->bundle.' identified by '.$this->uuid);

        $response = $this->next($this->request);

        if ($response->status() !== 200) {
            Log::info('[LHG Commons] TranslateRestoredContent: The content has not been saved due to a formular error, skipping the translation');

            return $response;
        }

        $source = type_to_class($this->bundle)::query()->with($this->relations)->findOrFail($this->uuid);
        $this->currentLangcode = $source->langcode;
        $relations = collect();

        Log::info('[LHG Commons] TranslateRestoredContent: The original content has been restored, build the translation');

        foreach ($this->relations as $relation) {
            $data = data_get($source, $relation);

            $relations->put(Str::snake($relation), $data instanceof Collection ? $data->pluck('uuid')->toArray() : $data->uuid);
        }

        $data = collect($source->getAttributes())
            ->put('langcode', $this->getTargetLangcode())
            ->put('update_author', auth()->user()->uuid)
            ->put('uuid', $source->uuid)
            ->put('bundle', $this->bundle)
            ->put('type', 'node')
            ->except('pid', 'last_update', 'author_id')
            ->merge($relations->map(fn (string|array $uuids, string $type) => DuplicateLogicService::getEveryTranslatedContent(
                type:  data_get(Schema::getFieldOptions($this->bundle, $type), 'target.0', $type),
                uuids: $uuids,
                langcode: $this->getTargetLangcode())
            ))
            ->when(isset($source->section_content), fn ($data) => $data->put('content', DuplicateLogicService::generateFreshSections($source->section_content)));

        Log::info('[LHG Commons] TranslateRestoredContent: Restored translation data is ready, dispatching job to update the content');
        UpdateContentThroughJob::dispatch($this->bundle, $data->toArray())->onQueue(get_high_priority_queue_name());

        return $response;
    }
}
