<?php

namespace Inside\LHG\Services;

use Exception;
use GuzzleHttp\Client;
use Illuminate\Support\Collection;
use Inside\LHG\Exceptions\RightApiNotFoundException;
use Inside\LHG\Exceptions\RightApiUnauthoziredException;
use Inside\LHG\Facades\RightApi;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\HttpFoundation\Response;

class RightApiService
{
    private Client $client;

    public function __construct(
        private string $endpoint,
        private string $apiKey,
    ) {
    }

    private function getClient(): Client
    {
        return $this->client ??= new Client(['base_uri' => trim($this->endpoint, '/')]);
    }

    private function transformUser(array $user): array
    {
        $user['hotelCode'] = $user['hotel']['code'] ?? null;
        $user['hotelName'] = $user['hotel']['name'] ?? null;
        $user['brands'] = collect($user['brands'])->pluck('childCode')->toArray();
        $user['countries'] = collect($user['countries'])->pluck('childCode')->toArray();
        $user['managementModes'] = collect($user['managementModes'])->pluck('childCode')->toArray();
        $user['profiles'] = collect($user['profiles'])->pluck('childCode')->toArray();
        $user['businessUnits'] = collect($user['businessUnits'])->pluck('childCode')->toArray();
        $user['regionalOffices'] = collect($user['regionalOffices'])->pluck('childCode')->toArray();
        unset($user['hotel']);

        return $user;
    }

    private function get(string $uri): ResponseInterface
    {
        try {
            return $this->getClient()->get($uri, [
                'headers' => [
                    'Authorization' => 'ApiKey '.$this->apiKey,
                    'Accept' => 'application/json',
                ],
                'debug' => config('app.debug'),
            ]);
        } catch (Exception $e) {
            throw match ($e->getCode()) {
                Response::HTTP_NOT_FOUND => new RightApiNotFoundException("[Right API] Not found: $uri"),
                Response::HTTP_UNAUTHORIZED => new RightApiUnauthoziredException("[Right API] Unauthorized: $uri"),
                default => $e,
            };
        }
    }

    public function getBrands(): Collection
    {
        $response = $this->get('/api/application/brands');

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result)->map(fn ($brands) => [
            'code' => trim($brands['code']),
            'active' => $brands['active'],
        ]);
    }

    public function getBusinessUnits(): Collection
    {
        $response = $this->get('/api/application/bu');

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result)->map(fn ($businessUnit) => [
            'code' => trim($businessUnit['code']),
            'active' => $businessUnit['active'],
        ]);
    }

    public function getCountries(): Collection
    {
        $take = $this->getCountriesCount();

        $response = $this->get(sprintf('/api/application/countries?Take=%d&Skip=%d', $take, 0));

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result['items'])->map(fn ($countries) => [
            'code' => trim($countries['code']),
            'active' => $countries['active'],
        ]);
    }

    public function getCountriesCount(): int
    {
        $response = $this->get(sprintf('/api/application/countries?Take=%d&Skip=%d', 1, 0));

        $result = json_decode($response->getBody()->getContents(), true);

        return $result['totalCount'] ?? 0;
    }

    public function getManagementModes(): Collection
    {
        $response = $this->get('/api/application/managementmodes');

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result)->map(fn ($managementModes) => [
            'code' => trim($managementModes['code']),
            'active' => $managementModes['active'],
        ]);
    }

    public function getProfiles(): Collection
    {
        $response = $this->get('/api/application/profiles');

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result)->map(fn ($profile) => [
            'code' => trim($profile['code']),
            'active' => $profile['active'],
        ]);
    }

    public function getRegionalOffices(): Collection
    {
        $response = $this->get('/api/application/regionaloffices');

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result)->map(fn ($regionalOffices) => [
            'code' => trim($regionalOffices['code']),
            'active' => $regionalOffices['active'],
        ]);
    }

    public function getServices(): Collection
    {
        $response = $this->get('/api/application/services');

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result)->map(fn ($services) => [
            'code' => trim($services['code']),
            'active' => $services['active'],
        ]);
    }

    public function getUsers(int $take = 10, int $skip = 0): Collection
    {
        $response = $this->get(sprintf('/api/application/users?Take=%d&Skip=%d', $take, $skip));

        $result = json_decode($response->getBody()->getContents(), true);

        return collect($result['items'])->map(fn ($user) => $this->transformUser($user));
    }

    public function getUsersCount(): int
    {
        $response = $this->get(sprintf('/api/application/users?Take=%d&Skip=%d', 1, 0));

        $result = json_decode($response->getBody()->getContents(), true);

        return $result['totalCount'] ?? 0;
    }

    public function getUser(string $id): array
    {
        $response = $this->get(sprintf('/api/application/users/%s', $id));

        return $this->transformUser(json_decode($response->getBody()->getContents(), true));
    }

    public function getNotifiableUsers(?string $filter = null): Collection
    {
        $total = $this->getUsersCount();
        $page = ceil($total / 100);

        $users = collect();

        for ($i = 0; $i < $page; $i++) {
            $users = $users->merge($this->getUsers(100, $i * 100));
        }

        return $users->filter(
            fn ($user) => match ($filter) {
                RightApi::NEWSLETTER => in_array($user['notification'], [RightApi::NEWSLETTER, RightApi::BOTH]),
                RightApi::NOTIFICATION => in_array($user['notification'], [RightApi::NOTIFICATION, RightApi::BOTH]),
                RightApi::BOTH => in_array($user['notification'], [RightApi::BOTH]),
                default => ! is_null($user['notification']) && ! empty($user['notification']),
            }
        )->pluck('notification', 'userName');
    }
}
