<?php

namespace Inside\LHG\Services;

use Illuminate\Support\Collection;
use Inside\Content\Models\Content;

final class DuplicateLogicService
{
    public static function generateFreshSections(Collection $sections): array
    {
        return $sections->map(function ($section) {
            $bundle = str_replace('inside_section_', '', $section->getTable());

            return collect($section)
                ->put('bundle', $bundle)
                ->put('pgID', '_fresh')
                ->except('uuid', 'uuid_host')
                ->when(isset($section->section_content), fn ($current) => $current->put(
                    'content', self::generateFreshSections($section->section_content)
                ))
                ->when($bundle === 'invite', fn (Collection $current) => $current
                    ->put('date', get_date(data_get($current, 'date', now()))?->format('Y-m-d H:i:s'))
                    ->put('end_date', get_date(data_get($current, 'end_date', now()))?->format('Y-m-d H:i:s'))
                )
                ->toArray();
        })->toArray();
    }

    public static function getEveryTranslatedContent(string $type, string|array $uuids, string $langcode): array
    {
        return type_to_class($type)::query()
            ->select('uuid', 'uuid_host')
            ->whereIn('uuid', is_string($uuids) ? [$uuids] : $uuids)
            ->get()
            ->map(fn ($content) => self::getTranslatedContent($content, $langcode))
            ->pluck('uuid')
            ->toArray();
    }

    public static function getTranslatedContent(Content $content, string $langcode): ?Content
    {
        if ($content->content_type === 'users') {
            return $content;
        }

        $translation = $content->getTranslationIfExist($langcode);

        return $translation->langcode === $langcode ? $translation : null;
    }

    public static function getConfiguration(): array
    {
        return config('louvrehotels.autotranslation');
    }

    public static function isAutoTranslationDisabled(): bool
    {
        return ! data_get(static::getConfiguration(), 'enabled', false);
    }

    public static function isNotDuplicable(string $bundle): bool
    {
        return ! in_array($bundle, data_get(static::getConfiguration(), 'contentTypes', []));
    }
}
