<?php

namespace Inside\LHG\Actions;

use Closure;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContent;
use Inside\LHG\Jobs\TranslateArchiveJob;
use Inside\LHG\Jobs\UpdateContentThroughJob;
use Inside\LHG\Services\DuplicateLogicService;

final class TranslateContent
{
    protected string $bundle;

    protected string $class;

    protected array $relations;

    protected array $files;

    public function __construct(
        protected Request $request,
        private Closure $next
    ) {
    }

    public function handle(): JsonResponse
    {
        if (DuplicateLogicService::isAutoTranslationDisabled()) {
            return $this->next($this->request);
        }

        $this->bundle = str($this->request->getPathInfo())->after('content/')->before('/')->toString();

        if (! Schema::isContentType($this->bundle)) {
            return $this->next($this->request);
        }

        $this->prepare();

        if (DuplicateLogicService::isNotDuplicable($this->bundle)) {
            return $this->next($this->request);
        }

        return $this->duplicate();
    }

    /**
     * Do not use ->bundle. When it's for comments, the bundle represents the content type of the parent content.
     *
     * @return void
     */
    protected function prepare(): void
    {
        $this->class = type_to_class($this->bundle);

        $relations = collect(Schema::getRelations($this->bundle))->diff('authors')->values()->toArray();

        $this->relations = $this->request->only($relations);
        $this->files = Schema::getFieldListingOfType($this->bundle, ['file', 'image']);
    }

    protected function getCurrentLangcode(): string
    {
        return $this->request->get('langcode') ?? auth()->user->langcode ?? config('app.locale');
    }

    protected function getTargetLangcode(): string
    {
        return collect(list_languages())->reject($this->getCurrentLangcode())->first();
    }

    protected function next(Request $request): mixed
    {
        return ($this->next)($request);
    }

    protected function duplicate(): JsonResponse
    {
        Log::info('[LHG Commons] TranslateContent: Starting translation process');

        $response = $this->next($this->request);

        if ($response->status() !== 200) {
            Log::info('[LHG Commons] TranslateContent: Error during original content saving as invalid field, translation skipped');

            return $response;
        }

        Log::info('[LHG Commons] TranslateContent: Original content saved, starting translation in '.$this->getTargetLangcode());
        $uuid = blank($response->original['translations']) ? $response->original['uuid'] : $response->original['translations'][$this->getCurrentLangcode()]['uuid'];
        $source = $this->class::find($uuid);
        $langcode = collect(list_languages())->reject($source->langcode)->first();

        $data = collect($this->request->all())
            ->diff($this->relations)
            ->except('fields', 'wysiwyg_images')
            ->put('langcode', $this->getTargetLangcode())
            ->put('update_author', auth()->user()->uuid)
            ->put('uuid', $source->uuid)
            ->merge(collect($this->relations)->filter()->map(fn (string|array $uuids, string $type) => DuplicateLogicService::getEveryTranslatedContent(
                type:  data_get(Schema::getFieldOptions($this->bundle, $type), 'target.0', $type),
                uuids: $uuids,
                langcode: $this->getTargetLangcode())
            ))
            ->merge(collect($this->files)->mapWithKeys(fn (string $field) => [$field => $source->{$field}]))
            ->when(isset($source->section_content), fn ($data) => $data->put('content', DuplicateLogicService::generateFreshSections($source->section_content)));

        Log::info('[LHG Commons] TranslateContent: Translation data ready, dispatching job to update the content');
        UpdateContentThroughJob::dispatch($source->content_type, $data->toArray())->onQueue(get_high_priority_queue_name())->chain([
            new TranslateArchiveJob($source->content_type, $source, $langcode),
        ]);

        return $response;
    }
}
