<?php

namespace Inside\ISSF\Console;

use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Inside\Host\Bridge\BridgeContent;
use Inside\Twitter\Helpers\TwitterFeeder;

/**
 * Import social feeds into inside_socials.
 */
class ImportSocialFeed extends BaseImport
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'import:social-feed';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import social feeds into inside_socials';

    public function handle(): void
    {
        date_default_timezone_set('UTC');

        $model = 'Inside\Content\Models\Contents\SocialFeedAccount';
        if (! class_exists($model)) {
            $this->error('Missing content type "SocialFeedAccount". Please create a content type "social_feed_account" with field "title", "feed_type", "name" and "avatar_url"');

            return;
        }

        // Fetch accounts
        $query = call_user_func($model.'::withoutGlobalScopes');
        $accounts = $query->get();

        // Add new feed from accounts
        foreach ($accounts as $account) {
            $account->feed_type = trim($account->feed_type);
            $account->feed_type = Str::studly($account->feed_type);

            $fct = 'load'.$account->feed_type;
            if (method_exists($this, $fct)) {
                $this->info('Importing '.$account->feed_type.' '.$account->title);
                call_user_func_array([$this, 'load'.$account->feed_type], [$account]);
            } else {
                $this->error('Cannot import account '.$account->title.', unrecognized type '.$account->feed_type);
            }
        }

        // End
        $this->info('Done importing');
        $this->info($this->formatStats());
    }

    /**
     * @param mixed $account
     * @return void
     * @throws Exception
     */
    protected function loadTwitter($account): void
    {
        // Select last tweets (max 1000)
        $url = 'https://api.twitter.com/1.1/statuses/user_timeline.json';
        $url .= '?screen_name='.urlencode($account->title);
        $url .= '&count=1000';
        $url .= '&exclude_replies=true';
        $url .= '&tweet_mode=extended';

        // Last loaded tweet
        $last = DB::table('inside_socials')
            ->select('import_id')
            ->where('account_name', $account->title)
            ->where('account_type', $account->feed_type)
            ->orderBy('date', 'desc')
            ->limit(1)
            ->get();
        if (! $last->isEmpty()) {
            $url .= '&since_id='.$last->first()->import_id;
        }

        // Load
        $feeder = new TwitterFeeder();
        $tweets = $feeder->feed($url, []);

        // Save tweets to database
        foreach ($tweets as $tweet) {
            $tweet['text'] = trim($tweet['text']);

            // Add medias
            if (! empty($tweet['extended_entities']['media'])) {
                foreach ($tweet['extended_entities']['media'] as $media) {
                    $tweet['text'] .= '<a href="'.$media['url'].'" target="_blank" class="'.$media['type'].'"><img src="'.$media['media_url_https'].'" /></a>';
                }
            }

            DB::table('inside_socials')->insert([
                'account_name' => $account->title,
                'account_type' => $account->feed_type,
                'import_id' => $tweet['id_str'],
                'date' => date('Y-m-d H:i:s', strtotime($tweet['created_at'])),
                'content' => $tweet['text'],
            ]);
            $this->addStat($account->title, 'tweets');
        }

        // Update name + avatar
        if (count($tweets) > 0) {
            $bridge = new BridgeContent();

            $bridge->contentUpdate('social_feed_account', [
                'uuid' => $account->uuid,
                'type' => 'node',
                'bundle' => 'social_feed_account',
                'name' => $tweets[0]['user']['name'],
                'avatar_url' => $tweets[0]['user']['profile_image_url_https'],
            ]);
        }
    }

    /*
    protected function loadFacebook($account)
    {
        $token = '&access_token=' . getenv('FACEBOOK_TOKEN');
        $datas = [];
        $name = null;
        $picture = null;

        $url = "https://graph.facebook.com/v4.0/me?fields=name,picture,posts";
        $filter = "{description,created_time,link,message,picture,attachments{media_type,unshimmed_url}}";

        // Last loaded posts
        $last = DB::table('inside_socials')
            ->select('import_id')
            ->where('account_name', $account->title)
            ->where('account_type', $account->feed_type)
            ->orderBy('date', 'desc')
            ->limit(1)
            ->get();

        // filter posts from the last post date
        if ($last && !$last->isEmpty()) {
            $url .= '.since(' . strtotime($last->first()->datetime) . ')';
        }
        $url .= $filter;
        $url .= $token;

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);

        // loop on each pages of posts and add them to an array
        do {
            curl_setopt($curl, CURLOPT_URL, $url);
            $encodedDatas = curl_exec($curl);
            $pagedatas = json_decode($encodedDatas);
            (array)$datas += (array)$pagedatas->posts->data;
            $url = $pagedatas->posts->paging->next;
            if (!$name || !$picture) {
                $name = $pagedatas->name;
                $picture = $pagedatas->picture->data->url;
            }
        } while (!empty($pagedatas->posts->data) && $url);


        foreach ($datas as $post) {
            $post->message = trim($post->message);


            // Add medias
            if (!empty($post->attachments->data)) {
                foreach ($post->attachments->data as $media) {
                    if ($post->message) {
                        $post->message .= ' ';
                    }
                    $post->message .= '<a href="' . $media->unshimmed_url . '" target="_blank" class="' . $media->media_type . '"><img src="' . $media->unshimmed_url . '" /></a>';
                }
            }

            DB::table('inside_socials')->insert([
                'account_name' => $account->title,
                'account_type' => $account->feed_type,
                'import_id' => $post->id,
                'date' => date('Y-m-d H:i:s', strtotime($post->created_time)),
                'content' => $post->message,
            ]);
            $this->addStat($account->title, 'posts');
        }

        if (sizeof($datas) > 0) {
            $account->name = $name;
            $account->avatar_url = $picture;
            $account->save();
        }

    }*/
}
