<?php

namespace Inside\ISOD\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Ideas;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Channels\DatabaseChannel;
use Inside\Notify\Messages\MailMessage;
use Inside\Notify\Models\NotificationType;

/**
 * Class SimpleNotification
 *
 * A simple fast notification sent both on email and database channels
 *
 *
 * @package Inside\ISOD\Notifications
 */
class SimpleNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /** @var Ideas */
    public $idea;

    /** @var string */
    public $notificationName;

    /** @var User */
    public $user;

    public function __construct(Ideas $idea, ?User $user = null)
    {
        $this->idea             = $idea;
        $reflection             = new \ReflectionClass($this);
        $this->notificationName = Str::snake($reflection->getShortName());
        if ($user === null) {
            /** @var User $user */
            $user = Auth::user();
        }
        $this->user             = $user;
        Log::debug('[SimpleNotification] {'.$this->notificationName.'}');
    }

    /**
     * @param mixed $notifiable
     * @return array
     */
    public function via($notifiable): array
    {
        return ['mail', DatabaseChannel::class];
    }

    protected function getUrl(bool $absolute = true): string
    {
        $baseUrl = config('app.url') .'/';
        return ($absolute ? $baseUrl : '') . 'les-idees#&modal/ideas/' . $this->idea->uuid;
    }

    /**
     * @param mixed $notifiable
     * @return MailMessage
     */
    public function toMail($notifiable): MailMessage
    {
        $recipient  = Users::find($notifiable->uuid);
        $data       = [];
        $data['to'] = [
            'uuid'      => $recipient->uuid,
            'lastname'  => $recipient->lastname,
            'firstname' => $recipient->firstname,
            'fullname'  => $recipient->firstname . ' ' . $recipient->lastname,
            'image'     => $recipient->image,
        ];
        $author     = Users::find($this->user->uuid);
        if ($author) {
            $data['authors'] = [
                'uuid'      => $author->uuid,
                'lastname'  => $author->lastname,
                'firstname' => $author->firstname,
                'fullname'  => $author->firstname . ' ' . $author->lastname,
                'image'     => $author->image,
            ];
        }
        $data['model'] = $this->idea->toArray();
        $data          = array_map('trim', Arr::dot($data)); // Remove \n at end

        $subject    = Lang::get(
            'notifications.isod.email.' . $this->notificationName . '.subject',
            $data,
            $notifiable->langcode
        );
        $content    = Lang::get(
            'notifications.isod.email.' . $this->notificationName . '.content',
            $data,
            $notifiable->langcode
        );
        $buttonText = Lang::get(
            'notifications.isod.email.' . $this->notificationName . '.button',
            $data,
            $notifiable->langcode
        );

        $contents = explode("\n", $content);

        $mailMessage = (new MailMessage())->markdown(
            'notifications::email',
            ['locale' => $notifiable->langcode]
        )->model($this->idea)->subject('["' . env('APP_NAME') . '"] ' . $subject);

        // Intro lines
        foreach ($contents as $content) {
            $mailMessage->line($content);
        }

        // Add action
        $mailMessage->action(Lang::get($buttonText, [], $notifiable->langcode), $this->getUrl());

        return $mailMessage;
    }

    /**
     * @param mixed $notifiable
     * @return array
     */
    public function toDatabase($notifiable): array
    {
        /** @var NotificationType $notificationType */
        $notificationType = NotificationType::where('action', 'ideabox')->firstOrFail();

        return [
            'notification_type_id' => $notificationType->id,
            'type'                 => $notificationType->type,
            'parent'               => null,
            'user_uuid'            => $notifiable->uuid,
            'notifiable_type'      => Ideas::class,
            'notifiable_uuid'      => $this->idea->uuid,
            'notifiable_langcode'  => $notifiable->langcode,
            'data'                 => [
                'notification_name' => $this->notificationName,
                'from'              => $this->user->uuid,
                'url'               => $this->getUrl(false),
            ],
        ];
    }
}
