<?php

namespace Inside\ISOD\Listeners;

use Inside\ISOD\Events\CompletedEvent;
use Inside\ISOD\Events\EnteredEvent;
use Inside\ISOD\Events\EnterEvent;
use Inside\ISOD\Events\GuardEvent;
use Inside\ISOD\Events\LeaveEvent;
use Inside\ISOD\Events\TransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Workflow\Event\Event;
use Symfony\Component\Workflow\Event\GuardEvent as WorkflowGuardEvent;

/**
 * Class WorkflowEventSubscriber
 *
 * Convert Symfony events to our ISOD events
 *
 * @package Inside\ISOD\Listeners
 */
class WorkflowEventSubscriber implements EventSubscriberInterface
{
    /**
     * Special guard on reject
     *
     * Note: only complexe ideas may be rejected
     *
     * @param \Symfony\Component\Workflow\Event\GuardEvent $event
     */
    public function guardReject(WorkflowGuardEvent $event): void
    {
        $idea = $event->getSubject();
        if ($idea && $idea->idea_type == 'simple') {
            $event->setBlocked(true);
        }
    }

    /**
     * Special guard on work_in_progress
     *
     * Note: only complexe ideas may be rejected
     *
     * @param \Symfony\Component\Workflow\Event\GuardEvent $event
     */
    public function guardValidate(WorkflowGuardEvent $event): void
    {
        $idea = $event->getSubject();
        if ($idea && $idea->idea_type == 'simple') {
            $event->setBlocked(true);
        }
    }

    /**
     * Special guard on simple_work_in_progress
     *
     * Note: only complexe ideas may be rejected
     *
     * @param \Symfony\Component\Workflow\Event\GuardEvent $event
     */
    public function guardSimpleValidateProgress(WorkflowGuardEvent $event): void
    {
        $idea = $event->getSubject();
        if ($idea && $idea->idea_type == 'complexe') {
            $event->setBlocked(true);
        }
    }

    public function guardEvent(WorkflowGuardEvent $event): void
    {
        $workflowName   = $event->getWorkflowName();
        $transitionName = $event->getTransition()->getName();

        event(new GuardEvent($event));

        event('workflow.guard', $event);
        event(sprintf('workflow.%s.guard', $workflowName), $event);
        event(sprintf('workflow.%s.guard.%s', $workflowName, $transitionName), $event);
    }

    public function leaveEvent(Event $event): void
    {
        $places       = $event->getTransition()->getFroms();
        $workflowName = $event->getWorkflowName();

        event(new LeaveEvent($event));

        event('workflow.leave', $event);
        event(sprintf('workflow.%s.leave', $workflowName), $event);
        foreach ($places as $place) {
            event(sprintf('workflow.%s.leave.%s', $workflowName, $place), $event);
        }
    }

    public function transitionEvent(Event $event): void
    {
        $workflowName   = $event->getWorkflowName();
        $transitionName = $event->getTransition()->getName();

        event(new TransitionEvent($event));

        event('workflow.transition', $event);
        event(sprintf('workflow.%s.transition', $workflowName), $event);
        event(sprintf('workflow.%s.transition.%s', $workflowName, $transitionName), $event);
    }

    public function enterEvent(Event $event): void
    {
        $places       = $event->getTransition()->getTos();
        $workflowName = $event->getWorkflowName();

        event(new EnterEvent($event));

        event('workflow.enter', $event);
        event(sprintf('workflow.%s.enter', $workflowName), $event);
        foreach ($places as $place) {
            event(sprintf('workflow.%s.enter.%s', $workflowName, $place), $event);
        }
    }

    public function enteredEvent(Event $event): void
    {
        $places       = $event->getTransition()->getTos();
        $workflowName = $event->getWorkflowName();

        event(new EnteredEvent($event));

        event('workflow.entered', $event);
        event(sprintf('workflow.%s.entered', $workflowName), $event);
        foreach ($places as $place) {
            event(sprintf('workflow.%s.entered.%s', $workflowName, $place), $event);
        }
    }

    public function completedEvent(Event $event): void
    {
        $workflowName   = $event->getWorkflowName();
        $transitionName = $event->getTransition()->getName();

        event(new CompletedEvent($event));

        event('workflow.completed', $event);
        event(sprintf('workflow.%s.completed', $workflowName), $event);
        event(sprintf('workflow.%s.completed.%s', $workflowName, $transitionName), $event);
    }

    public static function getSubscribedEvents()
    {
        /**
         * Symfony Workflow event examples
         *
         * event: guard leave transition enter entered competed
         *
         * workflow.[event]
         * workflow.[workflow_name].[event]
         * workflow.[workflow_name].[event].[place_name]
         *
         * workflow.transition
         * workflow.[workflow_name].transition
         * workflow.[workflow_name].transition.[transition_name]
         *
         *
         *
         *
         */
        return [
            'workflow.guard'                          => ['guardEvent'],
            'workflow.leave'                          => ['leaveEvent'],
            'workflow.transition'                     => ['transitionEvent'],
            'workflow.enter'                          => ['enterEvent'],
            'workflow.entered'                        => ['enteredEvent'],
            'workflow.completed'                      => ['completedEvent'],
            'workflow.ideasbox.guard.reject'          => ['guardReject'],
            'workflow.ideasbox.guard.simple_validate' => ['guardSimpleValidateProgress'],
            'workflow.ideasbox.guard.validate'        => ['guardValidate'],
        ];
    }
}
