<?php

namespace Inside\ISOD\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inside\Content\Models\Contents\Ideas;
use Inside\Content\Models\Contents\Users;
use Inside\ISOD\Events\CollaboratorSetEvent;
use Inside\ISOD\Facades\IdeasBox;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Permission\Facades\Permission;

class PrepareIdeasUpdate
{
    /**
     * @param Request $request
     * @param Closure $next
     * @return mixed
     * @throws \Exception
     */
    public function handle(Request $request, Closure $next)
    {
        $method = $request->getMethod();
        $path   = $request->path();
        $transition = null;
        $newStatus  = null;
        $workflow = null;
        $idea = null;
        if ($method === 'PUT'
            && preg_match(
                '#^api/v1/content/ideas/([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})$#iD',
                $path,
                $matches
            ) > 0
        ) {
            // Check permission
            if (!Permission::allowed('update', 'ideas', $matches[1])) {
                throw AuthorizationException::create('update', 'ideas', $matches[1]);
            }

            $idea       = Ideas::find($matches[1]);
            if ($idea) {
                $inputs           = $request->all();
                $inputs['fields'] = [
                    'uuid',
                ];

                if ($inputs['status'] != $idea->idea_status) {
                    // Check transition
                    $workflow = IdeasBox::getWorkflow()->get($idea);
                    $transition = IdeasBox::getTransition($idea, $inputs['status']);
                    Log::debug(
                        '[IdeaBox::getTransition] idea {' . $idea->title . '} try to change status from ['
                        . $idea->idea_status . '] to [' . $inputs['status'] . '] => {' . $transition . '}'
                    );
                    if (!$transition || !$workflow->can($idea, $transition)) {
                        Log::error(
                            '[ManageIdeaWorkflow::handle] Forbiden status changes {' . $idea->title
                            . '} try to change status from [' . $idea->idea_status . '] to ['
                            . $inputs['status'] . '] => {' . $transition . '}'
                        );
                    } else {
                        $newStatus = $inputs['status'];
                    }
                    unset($inputs['status']);
                }

                // Note: add author here or inside-content will change author to logged one ...
                $inputs['author'] = $idea->authors->uuid;

                $manager = $idea->authors->operationalManager->first();

                // Check collaborator
                if ((!$idea->ideaCollaborator || $idea->ideaCollaborator->isEmpty()
                        || ($manager
                            && $idea->ideaCollaborator->first()->uuid == $manager->uuid))
                    && isset($inputs['idea_collaborator'])
                    && is_string($inputs['idea_collaborator'])
                    && !empty($inputs['idea_collaborator'])
                ) {
                    $collaborator = Users::find($inputs['idea_collaborator']);
                    if ($collaborator) {
                        event(new CollaboratorSetEvent($idea, $collaborator));
                        if ($newStatus && $newStatus != 'work_in_progress') {
                            $transition = IdeasBox::getTransition($idea, 'work_in_progress');
                            if (!$transition || !$workflow->can($idea, $transition)) {
                                Log::error(
                                    '[ManageIdeaWorkflow::handle] Forbiden status changes {' . $idea->title
                                    . '} try to change status from [' . $idea->getOriginal('idea_status') . '] to ['
                                    . $idea->idea_status . '] => {' . $transition . '}'
                                );
                                throw new \Exception('Impossible save state');
                            }
                        }
                    }
                }
                $request = $request->replace($inputs);
            }
        }

        // Actually save the idea
        $response = $next($request);

        if ($transition !== null) {
            // We got a transition so apply it now !
            // Refresh our idea from database
            $idea->refresh();
            // We apply this to send events
            $workflow->apply($idea, $transition);
        }

        return $response;
    }
}
