<?php

namespace Inside\ISOD\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\ICFO\Facades\Game;
use Inside\ICFO\Game\Models\Player;
use Inside\ISOD\Game\Exports\InformationsExport;
use Inside\ISOD\Game\Models\Informations;
use Inside\Permission\Models\User;
use Laravel\Lumen\Routing\Controller;
use Maatwebsite\Excel\Facades\Excel;
use OpenApi\Annotations as OA;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Class GameController
 *
 * @OA\Tag(name="Jeu de lancement Sodern",description="
# Introduction

Il s'agit des services déstinés au jeu de lancement de l'intranet de Sodern

#Rappel sur le jeu et principes

Le jeu est constitué de 8 questions pour lesquelles chaque question a 4 choix de réponses possibles.
Le backoffice stock pour chaque joueur les 8 réponses données. C'est le front qui en déduit son état d'avancement.
Une réponse non donnée a la valeur ""null"".
")
 *
 * @package Inside\ISOD\Http\Controllers
 */
class GameController extends Controller
{
    /**
     * Informations on current user
     *
     * @OA\Schema(
     *           schema="GameInformations",
     *           type="object",
     *           @OA\Property(
     *                property="answer1",
     *                description="La réponse 1 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=3
     *              ),
     *           @OA\Property(
     *                property="answer2",
     *                description="La réponse 2 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=2
     *              ),
     *           @OA\Property(
     *                property="answer3",
     *                description="La réponse 3 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=1
     *              ),
     *           @OA\Property(
     *                property="answer4",
     *                description="La réponse 4 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=3
     *              ),
     *           @OA\Property(
     *                property="answer5",
     *                description="La réponse 5 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=2
     *              ),
     *           @OA\Property(
     *                property="answer6",
     *                description="La réponse 6 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=0
     *              ),
     *           @OA\Property(
     *                property="answer7",
     *                description="La réponse 7 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=null
     *              ),
     *           @OA\Property(
     *                property="answer8",
     *                description="La réponse 8 (0: A, 1: B, 2: C, 3: D)",
     *                type="integer",
     *                minimum=0,
     *                nullable=true,
     *                example=null
     *              )
     *   )
     *
     * @OA\Get(
     *      path="/game/informations",
     *      operationId="gameInformations",
     *      tags={"Jeu de lancement Sodern"},
     *      summary="Permet de récupérer les informations sur le jeu de lancement de l'utilisateur courant",
     *     @OA\Response(
     *          response=200,
     *          description="Les informations stockés sur le jeu pour le joueur courant",
     *          @OA\JsonContent(ref="#/components/schemas/GameInformations")
     *      ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     * )
     *
     * @return array
     */
    public function informations(): array
    {
        if (!Auth::check()) {
            abort(401);
        }

        /** @var \Inside\Authentication\Models\User $auth */
        $auth = Auth::user();

        try {
            $informations = Informations::where('user_uuid', $auth->uuid)->firstOrFail();
        } catch (ModelNotFoundException $e) {
            // If this user game information is not known yet, let's start a fresh one
            $informations = Informations::create(
                [
                    'user_uuid' => $auth->uuid,
                ]
            );
        }

        return [
            'answer1' => $informations->answer1,
            'answer2' => $informations->answer2,
            'answer3' => $informations->answer3,
            'answer4' => $informations->answer4,
            'answer5' => $informations->answer5,
            'answer6' => $informations->answer6,
            'answer7' => $informations->answer7,
            'answer8' => $informations->answer8,
        ];
    }

    /**
     * Informations on current user
     *
     * @OA\Put(
     *      path="/game/informations",
     *      operationId="gameInformationsUpdate",
     *      tags={"Jeu de lancement Sodern"},
     *      summary="Permet de mettre à jour les informations sur le jeu de lancement de l'utilisateur courant",
     *     @OA\RequestBody(
     *        description="Les nouvelles informations de jeu à **stocker**",
     *        required=true,
     *        @OA\JsonContent(ref="#/components/schemas/GameInformations",
     *          example="{
    ""answer1"": 3,
    ""answer2"": 2,
    ""answer3"": 1,
    ""answer4"": 3,
    ""answer5"": 2,
    ""answer6"": 0,
    ""answer7"": null,
    ""answer8"": null
    }"),
     *     ),
     *     @OA\Response(
     *          response=200,
     *          description="Les informations stockés sur le jeu pour le joueur courant après la mise-à-jour",
     *          @OA\JsonContent(ref="#/components/schemas/GameInformations")
     *      ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     * )
     *
     * @return array
     */
    public function update(Request $request): array
    {
        $this->validate(
            $request,
            [
                'answer1' => 'required|integer|min:0|max:3',
                'answer2' => 'integer|min:0|max:3|nullable',
                'answer3' => 'integer|min:0|max:3|nullable',
                'answer4' => 'integer|min:0|max:3|nullable',
                'answer5' => 'integer|min:0|max:3|nullable',
                'answer6' => 'integer|min:0|max:3|nullable',
                'answer7' => 'integer|min:0|max:3|nullable',
                'answer8' => 'integer|min:0|max:3|nullable',
            ]
        );

        /** @var \Inside\Authentication\Models\User $auth */
        $auth = Auth::user();

        try {
            $informations = Informations::where('user_uuid', $auth->uuid)->firstOrFail();
        } catch (ModelNotFoundException $e) {
            // If this user game information is not known yet, let's start a fresh one
            // Note: that should not happen because it's created before when front
            // asked for informations, but in case ... to avoid weird error
            $informations = Informations::create(
                [
                    'user_uuid' => $auth->uuid,
                ]
            );
        }
        $informations->answer1 = $request->get('answer1');
        $informations->answer2 = $request->get('answer2');
        $informations->answer3 = $request->get('answer3');
        $informations->answer4 = $request->get('answer4');
        $informations->answer5 = $request->get('answer5');
        $informations->answer6 = $request->get('answer6');
        $informations->answer7 = $request->get('answer7');
        $informations->answer8 = $request->get('answer8');

        $informations->save();

        return [
            'answer1' => $informations->answer1,
            'answer2' => $informations->answer2,
            'answer3' => $informations->answer3,
            'answer4' => $informations->answer4,
            'answer5' => $informations->answer5,
            'answer6' => $informations->answer6,
            'answer7' => $informations->answer7,
            'answer8' => $informations->answer8,
        ];
    }

    /**
     * Export
     *
     * @OA\Schema(
     *    schema="GameInformationsExport",
     *    type="object",
     *           @OA\Property(
     *                property="path",
     *                description="La chemin vers le fichier exporté",
     *                type="string",
     *                example="/storage/app/exports/20190716164409eae085e126f48073afe9c5e7eb8005e6_informations.xlsx"
     *           )
     *  )
     *
     * @OA\Get(
     *      path="/game/informations/export",
     *      operationId="gameInformationsExport",
     *      tags={"Jeu de lancement Sodern"},
     *      summary="Permet d'exporter les informations sur le jeu de lancement de tous les utilisateurs ayant terminé
     *      le jeu jusqu'au bout. Ceci est limité aux supers administrateurs.",
     *     @OA\Response(
     *          response=200,
     *          description="Le chemin de l'export ",
     *          @OA\JsonContent(ref="#/components/schemas/GameInformationsExport")
     *      ),
     *      @OA\Response(response=400, description="Bad request"),
     *      @OA\Response(response=401, description="Not allowed")
     * )
     *
     *
     * @return array
     * @throws AuthenticationException
     */
    public function export()
    {
        // Check super admin
        /** @var \Inside\Authentication\Models\User $auth */
        $auth = Auth::user();
        /** @var User $user */
        $user = User::find($auth->uuid);
        if (!$user->hasAnyRole(['super_administrator', 'administrator'])) {
            throw new AuthenticationException();
        }

        // Prepare export dir
        $exportsDir = cms_base_path(env('APP_STORAGE_PATH') . '/exports');

        if (File::exists($exportsDir) && !is_writable($exportsDir)) {
            throw new \Exception('Exports storage directory is not writable');
        }
        if (!File::exists($exportsDir)) {
            File::makeDirectory($exportsDir);
        }

        $fileName = Carbon::now()->format("YmdHis") . md5(Str::random(64)) . "_informations.xlsx";

        $path = 'exports/' . $fileName;

        if (!Excel::store(new InformationsExport(), $path)) {
            abort(400, 'failed to export data');
        }

        return [
            'path' => Config::get('app.url') . '/api/v1/game/export/' . $fileName,
        ];
    }

    public function serveExport(string $fileName): BinaryFileResponse
    {
        // Check super admin
        /** @var \Inside\Authentication\Models\User $auth */
        $auth = Auth::user();
        /** @var User $user */
        $user = User::find($auth->uuid);
        if (!$user->hasAnyRole(['super_administrator', 'administrator'])) {
            throw new AuthenticationException();
        }

        $disk = Storage::disk('local');
        $path = 'exports/' . $fileName;
        if (!$disk->exists($path)) {
            throw new ModelNotFoundException();
        }

        return new BinaryFileResponse($disk->path($path));
    }
}
