<?php

namespace Inside\ISOD\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Symfony\Component\Finder\Finder;

class UpdatePictures extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inside:isod:update-pictures';

    /**
     * Update user pictures
     *
     * @var string
     */
    protected $description = 'Update user pictures';

    /**
     * InsideImport constructor.
     *
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     *
     */
    public function handle(): void
    {
        $users = Users::query()->where('status', 1)->get();
        $bridge = new BridgeContent();

        $path = env('ISOD_PICTURES_REPLACE_PATH', '\\\\betelgeuse\\photos\\');

        if (!File::exists($path) || !File::isReadable($path)) {
            $this->warn(sprintf('Directory %s does not exist, check the ISOD_PICTURES_REPLACE_PATH environment variable', $path));
            return;
        }

        $this->info(sprintf('Handling %d users', count($users)));
        $bar = $this->output->createProgressBar(count($users));
        $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");
        $bar->start();

        $i = 0;
        $updated = 0;
        foreach ($users as $user) {
            $bar->setMessage(sprintf('Handling user %s [%d/%d]', $user->name, ++$i, count($users)));

            if (empty(trim($user->samaccountname))) {
                $bar->advance();
                continue;
            }

            $files = Finder::create()->in($path)->depth(0)->files()->name(
                '/^' . $user->samaccountname. '\.jpe?g/i'
            );

            if ($files->count() != 1) {
                $bar->advance();
                continue;
            }

            // We have exactly one file corresponding to our photo, get it !
            $iterator = $files->getIterator();
            $iterator->rewind();
            $photo = $iterator->current();

            // Read current photo
            $file = $photo->getContents();

            if (empty($file)) {
                $bar->advance();
                continue;
            }

            $chunkId    = Str::random(32);
            $chunkPath = sprintf('chunks/%s/%s.jpg', $chunkId, $user->samaccountname);
            Storage::makeDirectory(sprintf('chunks/%s', $chunkId));
            Storage::disk('local')->put($chunkPath, $file);

            $bridge->contentUpdate('users', [
                'uuid' => $user->uuid,
                'image' => $chunkPath
            ]);

            $updated++;

            $bar->advance();
        }

        $bar->finish();

        $this->info(sprintf("\n%d profile pictures were updated", $updated));
    }
}
